<?php

namespace App\Admin;

use App\Entity\Adherent;
use App\Entity\Cotisation;
use App\Entity\Geoloc;
use App\Entity\Groupe;
use App\Entity\User;
use App\Entity\Usergroup;
use App\Enum\MoyenEnum;
use FOS\UserBundle\Event\FilterUserResponseEvent;
use FOS\UserBundle\FOSUserEvents;
use Knp\Menu\ItemInterface as MenuItemInterface;
use Knp\Menu\ItemInterface;
use Sonata\AdminBundle\Admin\AbstractAdmin;
use Sonata\AdminBundle\Admin\AdminInterface;
use Sonata\AdminBundle\Datagrid\DatagridMapper;
use Sonata\AdminBundle\Datagrid\ListMapper;
use Sonata\AdminBundle\Form\FormMapper;
use Sonata\AdminBundle\Route\RouteCollection;
use Sonata\UserBundle\Model\UserManagerInterface;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Security;

/**
 * Administration des adhérents
 *
 * LOCO : Outil de gestion de Monnaie Locale Complémentaire
 * @author Julien Jorry <julien.jorry@gmail.com>
 */
class AdherentAdmin extends AbstractAdmin
{
    protected $baseRouteName = 'adherent';
    protected $baseRoutePattern = 'adherent';
    protected $security;

    protected $datagridValues = [
        // reverse order (default = 'ASC')
        '_sort_order' => 'DESC',
        // name of the ordered field (default = the model's id field, if any)
        '_sort_by' => 'updatedAt',
        // '_page' => 1,
        // '_per_page' => 32
    ];

    public function setSecurity(Security $security)
    {
        $this->security = $security;
    }

    public function configure()
    {
        parent::configure();
    }

    /**
    * {@inheritdoc}
    */
    public function createQuery($context = 'list')
    {
        $user = $this->security->getUser();
        $query = parent::createQuery($context);
        $query
            ->innerJoin($query->getRootAliases()[0] .'.user', 'u')
            ->addSelect('u')
        ;
        if ($user->isGranted('ROLE_GESTION_GROUPE') || $user->isGranted('ROLE_CONTACT')) {
            if (empty($user->getGroupesgere())) {
                $query->andWhere('false');
            } else {
                $groupe = $user->getGroupesgere();
                $query
                    ->andWhere($query->getRootAliases()[0] . '.groupe = :group')
                    ->setParameter('group', $groupe)
                ;
            }
        }
        return $query;
    }

    protected function configureSideMenu(MenuItemInterface $menu, $action, AdminInterface $childAdmin = null)
    {
        if (!$childAdmin && !in_array($action, ['edit', 'show'])) {
            return;
        }

        $admin = $this->isChild() ? $this->getParent() : $this;
        $id = $admin->getRequest()->get('id');
        $user = $this->getConfigurationPool()->getContainer()->get('doctrine')->getRepository(User::class)->findOneBy(array('adherent' => $id));

        if ($this->isGranted('EDIT') && $user != null) {
            $menu->addChild('Edit User', [
                'uri' => $this->getConfigurationPool()->getContainer()->get('router')->generate('admin_app_user_edit', ['id' => $user->getId()], UrlGeneratorInterface::ABSOLUTE_URL)
            ]);
        }
    }

    /**
    * {@inheritdoc}
    */
    protected function configureFormFields(FormMapper $formMapper): void
    {
        // Initialize adherent
        $adherent = $this->getSubject();
        $now = new \DateTime();
        if ($this->isCurrentRoute('create')) {
            $user = $this->userManager->createUser();
            $groupe = $this->getConfigurationPool()->getContainer()->get('doctrine')->getRepository(Usergroup::class)->findOneByName('Adherent');
            $user->setEnabled(true);
            $user->addGroup($groupe);
            $user->addRole('ROLE_ADHERENT');
            $adherent->setEcompte('0');
            $user->setAdherent($adherent);
            $adherent->setUser($user);
        }
        if (count($adherent->getUser()->getCotisations()) <= 0) {
            $cotisation = new Cotisation();
            $cotisation->setOperateur($adherent->getUser());
            $cotisation->setExpediteur($adherent);
            $cotisation->setDebut($now);
            $cotisation->setFin(new \DateTime('+ 1 year'));
            $adherent->getUser()->addCotisation($cotisation);
        }
        if ($adherent->getGeoloc() == null) {
            $adherent->setGeoloc(new Geoloc());
        }
        //nom, prénom, adresse, tel, mail et cotisation en une seule fois et générer un mdp
        $formMapper
            ->tab('General')
                ->with('Identité', ['class' => 'col-md-7'])
                    ->add('user.firstname', TextType::class, array(
                        'label' => 'Prénom :',
                        'required' => true
                    ))
                    ->add('user.lastname', TextType::class, array(
                        'label' => 'Nom :',
                        'required' => true
                    ));
        if (!$this->isCurrentRoute('create')) {
            $formMapper
                ->add('user.username', TextType::class, array(
                    'label' => 'Username :',
                    'required' => true,
                    'disabled' => true
                ));
        }
                $formMapper
                    ->add('user.phone', TextType::class, array(
                        'label' => 'Téléphone :',
                        'required' => false
                    ))
                    ->add('user.email', TextType::class, array(
                        'label' => 'Email :',
                        'required' => true
                    ))
                    ->add('groupe', ChoiceType::class, array(
                        'required' => true,
                        'label' => 'Groupe local :',
                        'choices' => $this->getConfigurationPool()->getContainer()->get('doctrine')->getRepository(Groupe::class)->findAll(),
                        'choice_label' => 'name',
                        'placeholder' => 'Choisir un groupe',
                    ))
                ->end()
                ->with('Cotisation', ['class' => 'col-md-5'])
                //@TODO : géré une ou plusieurs cotisations
                    ->add('user.cotisations.first.cotisationInfos.annee', TextType::class, array('label' => 'Année', 'data' => $now->format('Y')))
                    ->add('user.cotisations.first.montant', TextType::class, array('label' => 'Montant'))
                    ->add('user.cotisations.first.moyen', ChoiceType::class, array(
                        'required' => true,
                        'label' => 'Moyen :',
                        'choices' => MoyenEnum::getAvailableTypes(),
                        'choice_label' => function ($choice) {
                            return MoyenEnum::getTypeName($choice);
                        },
                    ));
        if ($this->security->getUser()->isGranted('ROLE_TRESORIER')) {
            $formMapper
                ->add('user.cotisations.first.cotisationInfos.recu', CheckboxType::class, array('label' => 'Reçu'));
        }
        $formMapper->end();
        if (!$this->isCurrentRoute('create')) {
            $formMapper
                ->with('Date', ['class' => 'col-md-5'])
                    ->add('user.cotisations.first.cotisationInfos.debut', DateType::class, array(
                        'label' => 'Date de début',
                        'data' => new \DateTime(),
                        'widget' => 'single_text',
                        'html5' => false,
                        'attr' => ['class' => 'js-datepicker'],
                    ))
                    ->add('user.cotisations.first.cotisationInfos.fin', DateType::class, array(
                        'label' => 'Date de fin',
                        'data' => new \DateTime('+ 1 year'),
                        'widget' => 'single_text',
                        'html5' => false,
                        'attr' => ['class' => 'js-datepicker'],
                    ))
                ->end();
        }
            $formMapper
                ->with('Addresse', ['class' => 'col-md-7'])
                    ->add('geoloc.adresse', TextType::class, array(
                        'label' => 'Addresse :',
                        'required' => true
                    ))
                    ->add('geoloc.cpostal', TextType::class, array(
                        'label' => 'Code postal :',
                        'required' => true
                    ))
                    ->add('geoloc.ville', TextType::class, array(
                        'label' => 'Ville :',
                        'required' => true
                    ))
                ->end()
            ->end()
        ;
        parent::configureFormFields($formMapper);
    }

    public function preUpdate($adherent)
    {
        $this->prePersist($adherent);
    }

    public function prePersist($adherent)
    {
        $em = $this->getConfigurationPool()->getContainer()->get('doctrine')->getManager();

        if (empty($adherent->getUser()->getUsername())) {
            $adherent->getUser()->setUsername($adherent->getUser()->getEmail());
        }
        if (empty($adherent->getUser()->getPassword())) {
            // $tokenGenerator = $this->getConfigurationPool()->getContainer()->get('fos_user.util.token_generator');
            // $password = substr($tokenGenerator->generateToken(), 0, 12);
            $bytes = random_bytes(64);
            $password = rtrim(strtr(base64_encode($bytes), '+/', '-_'), '=');
            $adherent->getUser()->setPassword($password);
            // @TODO : send email to user
        }
        $this->userManager->updateUser($adherent->getUser());
        $adherent->getUser()->createEmailToken();
        $em->persist($adherent->getUser());
        $em->persist($adherent);
        $em->flush();
        // @TODO : envoyer un mail au nouvel utilisateur avec l'emailtoken via le dispatch d'un event
        // $this->eventDispatcher->dispatch(FOSUserEvents::REGISTRATION_COMPLETED, new FilterUserResponseEvent($adherent->getUser(), $this->getRequest(), $response));
    }

    /**
    * {@inheritdoc}
    */
    protected function configureDatagridFilters(DatagridMapper $datagridMapper): void
    {
        $datagridMapper
            ->add('user.username', null, array('label' => 'Login'))
            ->add('user.email', null, array('label' => 'Email'))
        ;
    }

    /**
     * @param EventDispatcherInterface $userManager
     */
    public function setEventDispatcher(EventDispatcherInterface $eventDispatcher): void
    {
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @return EventDispatcherInterface
     */
    public function getEventDispatcher()
    {
        return $this->eventDispatcher;
    }

    /**
     * @param UserManagerInterface $userManager
     */
    public function setUserManager(UserManagerInterface $userManager): void
    {
        $this->userManager = $userManager;
    }

    /**
     * @return UserManagerInterface
     */
    public function getUserManager()
    {
        return $this->userManager;
    }

    protected function configureListFields(ListMapper $listMapper): void
    {
        unset($this->listModes['mosaic']);
        $listMapper
            ->addIdentifier('user.username', null, array('label' => 'Login'))
            ->addIdentifier('user.email', null, array('label' => 'Email'))
            ->addIdentifier('ecompte', null, array('label' => 'Ecompte'))
            ->addIdentifier('groupe', null, array(
                'label' => 'Groupe',
                'sortable' => true,
                'sort_field_mapping' => array('fieldName' => 'name'),
                'sort_parent_association_mappings' => array(array('fieldName' => 'groupe'))
            ))
            ->addIdentifier('user.enabled', null, array('label' => 'Activé', 'datatype' => 'App.User', 'template' => '@SonataAdmin/Boolean/editable_boolean.html.twig'))
            ->addIdentifier('user.updatedAt', null, array('label' => 'Mis à jour'))
        ;
    }

    protected function configureRoutes(RouteCollection $collection)
    {
        $collection->remove('delete');
    }

    public function getBatchActions()
    {
        $actions = parent::getBatchActions();
        unset($actions['delete']);

        return $actions;
    }
}
