<?php

/*
 * kohinos_cooperatic
 * Copyright (C) 2019-2020  ADML63
 * Copyright (C) 2020- Cooperatic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
 
namespace App\Twig;

use App\Entity\Comptoir;
use App\Entity\Flux;
use App\Entity\Groupe;
use App\Entity\Groupeprestataire;
use App\Entity\News;
use App\Entity\Prestataire;
use App\Entity\Rubrique;
use App\Entity\Siege;
use App\Entity\User;
use App\Entity\GlobalParameter;
use App\Util\WordpressUtil;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Security\Core\User\UserInterface;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;
use Twig\TwigTest;

class AppExtension extends AbstractExtension
{
    public $em;
    public $security;
    public $container;
    public $paginator;
    public $session;
    public $wp;

    public function __construct(ContainerInterface $container, Security $security, EntityManagerInterface $em, PaginatorInterface $paginator, SessionInterface $session, WordpressUtil $wp)
    {
        $this->em = $em;
        $this->security = $security;
        $this->container = $container;
        $this->paginator = $paginator;
        $this->session = $session;
        $this->wp = $wp;
    }

    public function getFunctions()
    {
        return [
            new \Twig_SimpleFunction('showModalGroupChoice', array($this, 'showModalGroupChoice')),
            new \Twig_SimpleFunction('getCurrentComptoir', array($this, 'getCurrentComptoir')),
            new \Twig_SimpleFunction('getCurrentGroupe', array($this, 'getCurrentGroupe')),
            new \Twig_SimpleFunction('getCurrentPrestataire', array($this, 'getCurrentPrestataire')),
            new \Twig_SimpleFunction('isCotisationValid', array($this, 'isCotisationValid')),
            new \Twig_SimpleFunction('getSiege', array($this, 'getSiege')),
            new \Twig_SimpleFunction('isDevFixture', array($this, 'isDevFixture')),
            new \Twig_SimpleFunction('getLastNews', array($this, 'getLastNews')),
            new \Twig_SimpleFunction('getAllPrestataires', array($this, 'getAllPrestataires')),
            new \Twig_SimpleFunction('getAllGroupePrestataires', array($this, 'getAllGroupePrestataires')),
            new \Twig_SimpleFunction('getAllComptoirs', array($this, 'getAllComptoirs')),
            new \Twig_SimpleFunction('getAllRubriques', array($this, 'getAllRubriques')),
            new \Twig_SimpleFunction('getAllGroupes', array($this, 'getAllGroupes')),
            new \Twig_SimpleFunction('getAllFlux', array($this, 'getAllFlux')),
            new \Twig_SimpleFunction('mediaurl', array($this, 'mediaurl')),
            new \Twig_SimpleFunction('getWordpressApiKey', array($this, 'getWordpressApiKey')),
            new \Twig_SimpleFunction('parameter', function ($name) {
                return $this->container->getParameter($name);
            }),
            new \Twig_SimpleFunction('isPayzenEnabled', array($this, 'isPayzenEnabled'))
        ];
    }

    public function isPayzenEnabled()
    {
        return $this->em->getRepository(GlobalParameter::class)->val(GlobalParameter::USE_PAYZEN) === 'true' ? true : false;
    }

    public function getFilters()
    {
        return array(
            new \Twig_SimpleFilter('json_decode', [$this, 'jsonDecode']),
            new \Twig_SimpleFilter('safe_email', [$this, 'safeEmailFilter']),
        );
    }

    public function jsonDecode($string)
    {
        return json_decode($string, true);
    }

    public function getWordpressApiKey()
    {
        return $this->wp->getApiKey();
    }

    public function showModalGroupChoice()
    {
        if ($this->security->getUser() != null) {
            if (count($this->security->getUser()->getPossiblegroups()) > 1 && count($this->security->getUser()->getGroups()) == 0) {
                return true;
            }
        }
        return false;
    }

    public function getCurrentComptoir()
    {
        if (!$this->session->has('_comptoirgere')) {
            return null;
        }
        return $this->em->getRepository(Comptoir::class)->findOneById($this->session->get('_comptoirgere')->getId());
    }

    public function getCurrentGroupe()
    {
        if (!$this->session->has('_groupegere')) {
            return null;
        }
        return $this->em->getRepository(Groupe::class)->findOneById($this->session->get('_groupegere')->getId());
    }

    public function getCurrentPrestataire()
    {
        if (!$this->session->has('_prestagere')) {
            return null;
        }
        return $this->em->getRepository(Prestataire::class)->findOneById($this->session->get('_prestagere')->getId());
    }

    public function isCotisationValid(): bool
    {
        if ($this->security->getUser() != null) {
            if ($this->security->getUser()->isGranted('ROLE_ADHERENT') && $this->security->getUser()->getAdherent() != null) {
                $query = $this->em->getRepository(Flux::class)->getQueryByAdherent($this->security->getUser()->getAdherent(), 'cotisation');
            } elseif ($this->security->getUser()->isGranted('ROLE_PRESTATAIRE') && $this->session->get('_prestagere') != null) {
                $query = $this->em->getRepository(Flux::class)->getQueryByPrestataire($this->session->get('_prestagere'));
            }
            $cotisations = $query->getResult();
            foreach ($cotisations as $cotisation) {
                if ($cotisation->getCotisationInfos()->getFin() > new \DateTime("now")) {
                    return true;
                }
            }
        }
        return false;
    }

    public function isDevFixture(?string $username = null)
    {
        if ($username == null) {
            $user = $this->em->getRepository(User::class)->findBy(array('username' => ['user_redacteur', 'user_tresorier', 'user_prestataire', 'user_contact']));

            return (count($user) === 4);
        }

        return ($this->em->getRepository(User::class)->findOneByUsername($username) != null);
    }

    public function getSiege()
    {
        return $this->em->getRepository(Siege::class)->findOneById(1);
    }

    public function getLastNews($limit = 5)
    {
        return $this->em->getRepository(News::class)->findBy(array('enabled' => true), array('createdAt' => 'DESC'), $limit);
    }

    public function getAllPrestataires()
    {
        return $this->em->getRepository(Prestataire::class)->findBy(array('mlc' => false, 'enabled' => true), array('raison'=> 'ASC'));
    }

    public function getAllGroupePrestataires($type = '')
    {
        return $this->em->getRepository(Groupeprestataire::class)->findBy(array('type' => $type, 'enabled' => true));
    }

    public function getAllComptoirs()
    {
        return $this->em->getRepository(Comptoir::class)->findBy(array('enabled' => true), array('name'=> 'ASC'));
    }

    public function getAllRubriques()
    {
        return $this->em->getRepository(Rubrique::class)->findBy(array('enabled' => true), array('name'=> 'ASC'));
    }

    public function getAllFlux(User $user, Request $request, $parenttype = null)
    {
        $query = null;
        if ($this->session->get('_prestagere') != null) {
            if ($this->security->isGranted('ROLE_PRESTATAIRE')) {
                $query = $this->em->getRepository(Flux::class)->getQueryByPrestataire($this->session->get('_prestagere'), $parenttype);
            } elseif ($this->security->isGranted('ROLE_CAISSIER_PRESTATAIRE')) {
                $query = $this->em->getRepository(Flux::class)->getQueryByCaissierprestataire($this->session->get('_prestagere'));
            }
        } elseif ($user->getAdherent() != null) {
            $query = $this->em->getRepository(Flux::class)->getQueryByAdherent($user->getAdherent(), $parenttype);
        } elseif ($this->session->get('_comptoirgere') != null && ($this->security->isGranted('ROLE_COMPTOIR') || $this->security->isGranted('ROLE_CAISSIER_COMPTOIR'))) {
            $query = $this->em->getRepository(Flux::class)->getQueryByComptoir($this->session->get('_comptoirgere'));
        } elseif ($this->session->get('_groupegere') != null) {
            $query = $this->em->getRepository(Flux::class)->getQueryByGroupe($this->session->get('_groupegere'));
        }
        if ($query != null) {
            $pagination = $this->paginator->paginate(
                $query, /* query NOT result */
                $request->query->getInt('page', 1)/*page number*/,
                10/*limit per page*/
            );

            return $pagination;
        }
        return null;
    }

    public function getAllGroupes()
    {
        return $this->em->getRepository(Groupe::class)->findBy(array('enabled' => true), array('name'=> 'ASC'));
    }

    public function mediaurl($media, $format)
    {
        $provider = $this->container->get($media->getProviderName());

        return $provider->generatePublicUrl($media, $format);
    }

    public function getTests(): array
    {
        return [
            new TwigTest('instanceof', [$this, 'instanceof']),
        ];
    }

    public function instanceof($var, $instance)
    {
        $reflexionClass = new \ReflectionClass($instance);
        ;

        return $reflexionClass->isInstance($var);
    }

    /**
     * Protects email address.
     * (inspired by : https://github.com/getgrav/grav/blob/develop/system/src/Grav/Common/Twig/TwigExtension.php )
     *
     * @param  string $str
     *
     * @return string
     */
    public function safeEmailFilter($str)
    {
        $email   = '';
        for ($i=0, $len = strlen($str); $i < $len; $i++) {
            $j = mt_rand(0, 1);
            if ($j === 0) {
                $email .= '&#'.ord($str[$i]).';';
            } elseif ($j === 1) {
                $email .= $str[$i];
            }
        }
        return str_replace('@', '&#64;', $email);
    }
}
