<?php

/*
 * kohinos_cooperatic
 * Copyright (C) 2019-2020  ADML63
 * Copyright (C) 2020- Cooperatic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
 
namespace App\Form\Type;

use App\Entity\Adherent;
use App\Entity\Cotisation;
use App\Entity\Groupe;
use App\Entity\Prestataire;
use App\Entity\User;
use App\Entity\GlobalParameter;
use App\Enum\MoyenEnum;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\GreaterThanOrEqual;

class CotiserFormType extends AbstractType
{
    protected $em;
    protected $security;
    protected $session;

    public function __construct(EntityManagerInterface $em, Security $security, SessionInterface $session)
    {
        $this->em = $em;
        $this->security = $security;
        $this->session = $session;
    }

    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        if (empty($this->security) || empty($this->security->getUser())) {
            throw new \Exception("Opération impossible ! Utilisateur non connecté !");
        }
        $now = new \DateTime();
        $montant = 0;
        $canPayWithMlc = false;
        if ($this->security->getUser()->isGranted('ROLE_ADHERENT')) {
            $montant = floatval($this->em->getRepository(GlobalParameter::class)->val(GlobalParameter::COTISATION_ADHERENT));
            $canPayWithMlc = ($this->security->getUser()->getAdherent()->getEcompte() >= $montant);
        } elseif ($this->security->getUser()->isGranted('ROLE_PRESTATAIRE')) {
            $montant = floatval($this->em->getRepository(GlobalParameter::class)->val(GlobalParameter::COTISATION_PRESTATAIRE));
            $canPayWithMlc = ($this->session->get('_prestagere')->getEcompte() >= $montant);
        }

        $canPayWithCB = $this->em->getRepository(GlobalParameter::class)->val(GlobalParameter::USE_PAYZEN) === 'true' ? true : false;

        $builder
            ->add('operateur', HiddenType::class, array(
                'entity_class' => User::class,
                'em' => $this->em,
                'data_class' => null,
                'data' => $this->security->getUser()->getId()
            ))
            ->add('role', HiddenType::class, array(
                'data' => $this->security->getUser()->getGroups()[0]->__toString()
            ))
            ->add('destinataire', HiddenType::class, array(
                'data' => $this->em->getRepository(Prestataire::class)->findOneBy(array('mlc' => true))->getId(),
                'data_class' => null,
                'entity_class' => Prestataire::class,
                'em' => $this->em
            ))
            ->add('reference', HiddenType::class, array(
                'label' => 'Reference :',
                'required' => true,
                'data' => 'Cotisation '.$now->format('Y')
            ))
            ->add('montant', IntegerType::class, array(
                'data' => $montant,
                'constraints' => [
                    new NotBlank(),
                    new GreaterThanOrEqual(['value' => $montant]),
                ],
            ))
            ->add('moyen', HiddenType::class, array(
                'data' => MoyenEnum::MOYEN_MLC,
            ))
        ;
        if ($canPayWithMlc) {
            $builder->add('payMLC', SubmitType::class, array(
              'label' => "Payer en eMLC",
              'attr' => [
                'class' => 'btn-primary btn cotisationMLCSubmit'
              ]
            ));
        }
        if ($canPayWithCB) {
            $builder->add('payCB', SubmitType::class, array(
              'label' => "Payer en CB",
              'attr' => [
                'class' => 'btn-primary btn cotisationCBSubmit'
              ]
            ));
        }

        if ($this->security->getUser()->isGranted('ROLE_ADHERENT')) {
            $builder
                ->add('expediteur', HiddenType::class, array(
                    'entity_class' => Adherent::class,
                    'em' => $this->em,
                    'data_class' => null,
                    'data' => $this->security->getUser()->getAdherent()->getId()
                ))
            ;
        } elseif ($this->security->getUser()->isGranted('ROLE_PRESTATAIRE')) {
            $builder
                ->add('expediteur', HiddenType::class, array(
                    'entity_class' => Prestataire::class,
                    'em' => $this->em,
                    'data_class' => null,
                    'data' => $this->session->get('_prestagere')->getId()
                ))
            ;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'cascade_validation' => true
        ));
    }

    public function getBlockPrefix()
    {
        return 'formCotiser';
    }
}
