<?php

/*
 * kohinos_cooperatic
 * Copyright (C) 2019-2020  ADML63
 * Copyright (C) 2020- Cooperatic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
 
namespace App\Admin;

use App\Entity\Comptoir;
use App\Entity\ContactComptoir;
use App\Entity\Geoloc;
use App\Entity\GlobalParameter;
use App\Entity\Groupe;
use App\Entity\User;
use App\Entity\Adherent;
use App\Entity\Usergroup;
use App\Form\Type\ContactEntityFormType;
use App\Form\Type\GeolocFormType;
use App\Form\Type\UserWithPswdFormType;
use Doctrine\ORM\EntityRepository;
use FOS\CKEditorBundle\Form\Type\CKEditorType;
use Sonata\AdminBundle\Admin\AbstractAdmin;
use Sonata\AdminBundle\Datagrid\DatagridMapper;
use Sonata\AdminBundle\Datagrid\ListMapper;
use Sonata\AdminBundle\Form\FormMapper;
use Sonata\AdminBundle\Route\RouteCollection;
use Sonata\AdminBundle\Show\ShowMapper;
use Sonata\DoctrineORMAdminBundle\Filter\CallbackFilter;
use Sonata\MediaBundle\Form\Type\MediaType;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\CollectionType;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormError;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\ArrayCollection;
use FOS\UserBundle\Model\UserManagerInterface;

/**
 * Administration des comptoirs
 *
 * KOHINOS : Outil de gestion de Monnaie Locale Complémentaire
 * @author Julien Jorry <julien.jorry@gmail.com>
 */
class ComptoirAdmin extends AbstractAdmin
{
    protected $security;
    protected $baseRouteName = 'comptoir';
    protected $baseRoutePattern = 'comptoir';
    protected $datagridValues = [
        '_sort_order' => 'ASC',
        '_sort_by' => 'name',
    ];

    public function setSecurity(Security $security)
    {
        $this->security = $security;
    }

    /**
    * {@inheritdoc}
    */
    public function createQuery($context = 'list')
    {
        $query = parent::createQuery($context);
        $user = $this->security->getUser();
        if (empty($this->getRequest()->getSession()->get('_groupegere'))) {
            // TODO: Pourquoi empêcher toutes les "requêtes comptoir" pour ces rôles ?
            //if ($user->isGranted('ROLE_GESTION_GROUPE') || $user->isGranted('ROLE_CONTACT') || $user->isGranted('ROLE_TRESORIER')) {
            if ($user->isGranted('ROLE_GESTION_GROUPE') || $user->isGranted('ROLE_CONTACT')) {
                $query->andWhere('false = true');
            }
        } else {
            $query
                ->andWhere($query->getRootAliases()[0].'.groupe = :groupe')
                ->setParameter('groupe', $this->getRequest()->getSession()->get('_groupegere'))
            ;
        }
        return $query;
    }

    /**
    * {@inheritdoc}
    */
    protected function configureFormFields(FormMapper $formMapper)
    {
        $comptoir = $this->getSubject();
        $user = $this->security->getUser();
        if ($this->isCurrentRoute('create')) {
            $geoloc = new Geoloc();
            $comptoir->setGeoloc($geoloc);
        }
        // get the current Image instance
        $imageHelp = null;
        if (!empty($comptoir) && !empty($comptoir->getMedia())) {
            $image = $comptoir->getMedia();
            if ($image && ($webPath = $image->getWebPath())) {
                // get the container so the full path to the image can be set
                $container = $this->getConfigurationPool()->getContainer();
                $fullPath = $container->get('request_stack')->getCurrentRequest()->getBasePath().'/'.$webPath;
                // add a 'help' option containing the preview's img tag
                $imageHelp = '<img src="'.$fullPath.'" class="admin-preview" />';
            }
        }
        $formMapper
            ->with('Comptoir', ['class' => 'col-md-6']);
        if (($user->isGranted('ROLE_GESTION_GROUPE') || $user->isGranted('ROLE_CONTACT')) && !empty($this->getRequest()->getSession()->get('_groupegere'))) {
            $formMapper->add('groupe', null, array(
                    'label' => 'Groupe local',
                    'required' => true,
                    'query_builder' => function (EntityRepository $repo) use ($user) {
                        return $repo->createQueryBuilder('g')
                            ->select('g')
                            ->join('g.gestionnaires', 'c')
                            ->where('c.id = :user')
                            ->setParameter('user', $user->getId());
                    },
                ));
        } else {
            $formMapper->add('groupe', null, array(
                    'label' => 'Groupe local',
                    'required' => true,
                ));
        }
        $formMapper
                ->add('name', null, array(
                    'label' => 'Nom',
                    'required' => true,
                ))
                ->add('content', CKEditorType::class, array(
                    'label' => 'Description :',
                    'required' => false
                ))
                ->add('enabled', null, array('label' => 'Activé ?'))
            ->end()
            ->with('Contact(s)', ['class' => 'col-md-6'])
                ->add('contacts', CollectionType::class, array(
                    'entry_type' => ContactEntityFormType::class,
                    'entry_options' => array('label' => false, 'data_class' => ContactComptoir::class),
                    'allow_add' => true,
                    'allow_delete' => true,
                    'by_reference' => false,
                    'label' => false
                ))
            ->end()
            ->with('Caissier(s)', ['class' => 'col-md-6'])
                ->add('caissiers', EntityType::class, [
                    // 'mapped' => false,
                    'class' => User::class,
                    'multiple' => true,
                    'required' => false,
                    'label' => 'Associer à un(des) utilisateur(s) existant :',
                    'choices' => $this->getConfigurationPool()->getContainer()->get('doctrine')->getRepository(User::class)->findAll(),
                    'choice_label' => 'username',
                    'placeholder' => 'Choisir un utilisateur',
                ])
                ->add('newcaissiers', CollectionType::class, [
                    'mapped' => false,
                    'required' => false,
                    'label' => 'Nouvel Utilisateur/Adhérent',
                    'entry_type' => UserWithPswdFormType::class,
                    'entry_options' => [
                        'required' => false,
                        'label' => false, ],
                    'by_reference' => false,
                    'allow_add' => true,
                    'allow_delete' => true,
                ])
            ->end() 
            ->with('Image', ['class' => 'col-md-6'])
                ->add('media', MediaType::class, array(
                    'provider' => 'sonata.media.provider.image',
                    'context' => 'prestataire',
                    'help' => $imageHelp,
                    'required' => false
                ))
            ->end()
            ->with('Geoloc', ['class' => 'col-md-6'])
                ->add('geoloc', GeolocFormType::class, array(
                    'label' => false,
                    'required' => true
                ))
            ->end()
        ;

        $em = $this->getConfigurationPool()->getContainer()->get('doctrine')->getManager();
        $formMapper->getFormBuilder()->addEventListener(FormEvents::POST_SUBMIT, function (FormEvent $event) use ($em) {
            $comptoir = $event->getData();
            $users = null;

            if (null != $event->getForm()->get('caissiers')->getData()) {
                $caissiers = $event->getForm()->get('caissiers')->getData();
                $this->addCaissers($caissiers, $comptoir);
            }
            if (null != $event->getForm()->get('newcaissiers')->getData()) {
                $newcaissiers = $event->getForm()->get('newcaissiers')->getData();
                $collection = new ArrayCollection($newcaissiers);
                $return = $this->addCaissers($collection, $comptoir);
                if (count($return) > 0) {
                    $event->getForm()->get('newcaissiers')->addError(new FormError('Caissiers : Courriel(s) déjà utilisé : ' . implode(', ', $return) . '!'));
                }
            }
        });
        parent::configureFormFields($formMapper);
    }

    private function addCaissers(Collection $users, Comptoir $comptoir)
    {
        $return = [];
        $em = $this->getConfigurationPool()->getContainer()->get('doctrine')->getManager();
        foreach ($users as $user) {
            if (null == $user->getId()) {
                $emailExist = $em->getRepository(User::class)->findBy(['email' => $user->getEmail()]);
                if (count($emailExist) > 0) {
                    $return[] = $emailExist;
                    break;
                }
                $user->setUsername($user->getEmail());
                $adh = new Adherent();
                $user->setAdherent($adh);
                $groupeAdh = $em->getRepository(Usergroup::class)->findOneByName('Adherent');
                $user->addPossiblegroup($groupeAdh);
            }

            $comptoir->addCaissier($user);
            $group = $em->getRepository(Usergroup::class)->findOneByName('Caissier de Comptoir');
            $user->addPossiblegroup($group);
            $this->userManager->updateUser($user);
            $em->persist($user);
            $em->persist($comptoir);
        }
        $em->flush();

        return $return;
    }

    /**
    * {@inheritdoc}
    */
    protected function configureDatagridFilters(DatagridMapper $datagridMapper): void
    {
        if ($this->isGranted('ROLE_ADMIN') || $this->isGranted('ROLE_SUPER_ADMIN') || $this->isGranted('ROLE_ADMIN_SIEGE')) {
            $datagridMapper
                ->add('groupe', null, [
                    'label' => "Groupe",
                    'show_filter' => true,
                    'advanced_filter' => false
                ])
            ;
        }
        $datagridMapper
            ->add('full_text', CallbackFilter::class, [
                'callback' => [$this, 'getFullTextFilter'],
                'field_type' => TextType::class,
                'label' => "Recherche par nom",
                'show_filter' => true,
                'advanced_filter' => false
            ])
        ;
    }

    public function getFullTextFilter($queryBuilder, $alias, $field, $value)
    {
        if (!$value['value']) {
            return;
        }

        // Use `andWhere` instead of `where` to prevent overriding existing `where` conditions
        $queryBuilder->andWhere(
            $queryBuilder->expr()->like($alias.'.name', $queryBuilder->expr()->literal('%' . $value['value'] . '%'))
        );

        return true;
    }

    /**
     * @param UserManagerInterface $userManager
     */
    public function setUserManager(UserManagerInterface $userManager): void
    {
        $this->userManager = $userManager;
    }

    /**
     * @return UserManagerInterface
     */
    public function getUserManager()
    {
        return $this->userManager;
    }

    protected function configureRoutes(RouteCollection $collection)
    {
        $collection->add('test');
        $collection->remove('delete');
    }

    public function configureActionButtons($action, $object = null)
    {
        $list = parent::configureActionButtons($action, $object);

        $list['test']['template'] = '@SonataAdmin/show_map_comptoir.html.twig';

        return $list;
    }
    
    public function getRouteShowOnFront($object)
    {
        return $this->routeGenerator->generate('show_comptoir', array('slug' => $object->getSlug()));
    }

    public function getRouteShowMap()
    {
        return $this->routeGenerator->generate('comptoirs_carte');
    }

    /**
    * {@inheritdoc}
    */
    protected function configureListFields(ListMapper $listMapper)
    {
        unset($this->listModes['mosaic']);

        $isWordpressActivated = $this->getConfigurationPool()->getContainer()->get('doctrine')->getRepository(GlobalParameter::class)->val(GlobalParameter::USE_WORDPRESS);
        if ($isWordpressActivated == 'false') {
            $actions = [
                'show' => ['template' => '@SonataAdmin/CRUD/list__action_showonfront.html.twig'],
                'edit' => []
            ];
        } else {
            $actions = [
                'edit' => []
            ];
        }

        $listMapper
            ->addIdentifier('name')
            ->addIdentifier('groupe', null, array(
                'label' => 'Groupe',
                'sortable' => true,
                'sort_field_mapping' => array('fieldName' => 'name'),
                'sort_parent_association_mappings' => array(array('fieldName' => 'groupe'))
            ))
            ->add('compte', null, array('label' => 'Solde'))
            ->add('geoloc', null, array('label' => 'Adresse'))
            ->add('enabled', null, array(
                'label' => 'Activé',
                'editable' => true
            ))
            ->add('_action', null, [
                'actions' => $actions
            ])
        ;
    }
    public function getExportFields()
    {
        return [
            'Id'                   => 'id',
            'Nom du comptoir'      => 'name',
            'Groupe'               => 'groupe',
            'Solde du comptoir'    => 'compte',
            'Adresse'              => 'geoloc',
            'Activé (1=oui)'       => 'enabled'
        ];
    }
}
