<?php
/**
 * AMAP class
 *
 * @package Default
 * @author
 **/
namespace App\Entity;

use App\Entity\EntityTrait\ContactEmailTelTrait;
use App\Entity\EntityTrait\EnablableEntityTrait;
use App\Entity\EntityTrait\GeolocEntityTrait;
use App\Entity\EntityTrait\NameSlugContentEntityTrait;
use App\Enum\GroupePrestaEnum;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity
 * @ORM\Table(name="groupeprestaire")
 */
class Groupeprestataire
{
    use NameSlugContentEntityTrait,
        TimestampableEntity,
        GeolocEntityTrait,
        ContactEmailTelTrait,
        EnablableEntityTrait;

    /**
     * @var int
     *
     * @ORM\Id
     * @ORM\Column(name="id", type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * @var string|null
     *
     * @ORM\Column(length=50)
     * @Assert\NotBlank
     * @Assert\Length(max=50)
     */
    protected $type;

    /**
     * @var ArrayCollection|Prestataire[]
     *
     * @ORM\ManyToMany(targetEntity="Prestataire", inversedBy="groupeprestataires", cascade={"persist"})
     */
    protected $prestataires;

    /**
     * @var Groupe $groupe
     *
     * @ORM\ManyToOne(targetEntity="Groupe", inversedBy="groupeprestataires")
     */
    private $groupe;

    /**
     * @var null|string (champ libre)
     *
     * @ORM\Column(name="horaires", type="string", length=255, nullable=true)
     */
    private $horaires;

    /**
    * @var \Application\Sonata\MediaBundle\Entity\Media
    * @ORM\ManyToOne(targetEntity="App\Application\Sonata\MediaBundle\Entity\Media", cascade={"persist"}, fetch="LAZY")
    * @ORM\JoinColumn(name="media_id", referencedColumnName="id")
    */
    protected $image;

    public function __construct()
    {
        $this->prestataires = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }

    /**
    * Get type
    * @return string
    */
    public function getType()
    {
        return $this->type;
    }

    /**
    * Set type
    * @return $this
    */
    public function setType($type)
    {
        if (!in_array($type, GroupePrestaEnum::getAvailableTypes())) {
            throw new \InvalidArgumentException("Type de groupe invalide [amap, marche] !");
        }
        $this->type = $type;

        return $this;
    }

    /**
    * Get horaires
    * @return string Horaires
    */
    public function getHoraires(): ?string
    {
        return $this->horaires;
    }

    /**
    * Set horaires
    * @return $this
    */
    public function setHoraires($horaires): self
    {
        $this->horaires = $horaires;
        return $this;
    }

    /**
     * @return Prestataire[]|ArrayCollection
     */
    public function getPrestataires()
    {
        return $this->prestataires;
    }

    /**
     * @param Prestataire $prestataire
     * @return $this
     */
    public function addPrestataire(Prestataire $prestataire): self
    {
        if (!$this->prestataires->contains($prestataire)) {
            $this->prestataires[] = $prestataire;
            $prestataire->addGroupeprestataire($this);
        }
        return $this;
    }

    /**
     * @param Prestataire $prestataire
     * @return $this
     */
    public function removePrestataire(Prestataire $prestataire): self
    {
        if ($this->prestataires->contains($prestataire)) {
            $this->prestataires->removeElement($prestataire);
            $prestataire->removeGroupeprestataire($this);
        }
        return $this;
    }

    /**
     * @param null|Groupe $groupes
     * @return $this
     */
    public function setGroupe(?Groupe $groupe)
    {
        $this->groupe = $groupe;
        return $this;
    }

    /**
     * @return null|Groupe
     */
    public function getGroupe(): ?Groupe
    {
        return $this->groupe;
    }

    /**
    * Get image
    * @return image|null
    */
    public function getImage()
    {
        return $this->image;
    }

    /**
    * Set image
    * @return $this
    */
    public function setImage($image)
    {
        $this->image = $image;
        return $this;
    }

    public function getPrestatairesCount()
    {
        return $this->getPrestataires()->count();
    }

    public function __toString(): string
    {
        $return = $this->getName() ? $this->getName() : 'GroupePrestataire';
        if ($this->type != null) {
            $return .= ' ('.strtoupper($this->type).')';
        }
        return $return;
    }
}
