<?php

namespace App\Admin;

use App\Entity\Adherent;
use App\Entity\Prestataire;
use App\Entity\Siege;
use App\Enum\MoyenEnum;
use Sonata\AdminBundle\Admin\AbstractAdmin;
use Sonata\AdminBundle\Datagrid\DatagridMapper;
use Sonata\AdminBundle\Datagrid\ListMapper;
use Sonata\AdminBundle\Form\FormMapper;
use Sonata\AdminBundle\Route\RouteCollection;
use Sonata\AdminBundle\Show\ShowMapper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateTimeType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\MoneyType;
use Symfony\Component\Intl\DateFormatter\IntlDateFormatter;
use Symfony\Component\Security\Core\Security;

/**
 * Administration des cotisations
 *
 * LOCO : Outil de gestion de Monnaie Locale Complémentaire
 * @author Julien Jorry <julien.jorry@gmail.com>
 */
class CotisationAdmin extends AbstractAdmin
{
    protected $baseRouteName = 'cotisation';
    protected $baseRoutePattern = 'cotisation';
    protected $security;
    protected $container;

    protected $translator;
    protected $datagridValues = [
        '_sort_order' => 'DESC',
        '_sort_by' => 'createdAt',
    ];

    public function setSecurity(Security $security, ContainerInterface $container)
    {
        $this->security = $security;
        $this->container = $container;
    }

    /**
    * {@inheritdoc}
    */
    protected function configureDatagridFilters(DatagridMapper $datagridMapper): void
    {
        $datagridMapper
            ->add('cotisationInfos.annee', null, array('label' => 'Année'))
            ->add('montant', null, array('label' => 'Montant'))
            ->add('cotisationInfos.recu', null, array('label' => 'Recu ?'))
        ;
    }

    /**
    * {@inheritdoc}
    */
    protected function configureFormFields(FormMapper $formMapper)
    {
        $cotisation = $this->getSubject();
        $now = new \DateTime();
        $formMapper
            ->with('Cotisation', ['class' => 'col-md-8'])
                ->add('parenttype', HiddenType::class, array(
                    'data' => 'cotisation'
                ))
                ->add('destinataire', HiddenType::class, array(
                    'data' => 1,
                    'data_class' => null,
                    'entity_class' => Siege::class,
                    'em' => $this->getConfigurationPool()->getContainer()->get('doctrine')->getManager()
                ))
                ->add('cotisationInfos.annee', null, array(
                    'label' => 'Année',
                ))
                ->add('montant', MoneyType::class, array(
                    'label' => 'Montant',
                    'data' => ($cotisation->getMontant() != null)?$cotisation->getMontant():floatval($this->container->getParameter('app.cotisation_montant'))
                ))
                ->add('moyen', ChoiceType::class, array(
                    'required' => true,
                    'choices' => MoyenEnum::getAvailableTypes(),
                    'choice_label' => function ($choice) {
                        return MoyenEnum::getTypeName($choice);
                    },
                ))
        ;
        if ($this->security->getUser() != null && $this->security->getUser()->isGranted('ROLE_TRESORIER')) {
            $formMapper
                ->add('cotisationInfos.recu', CheckboxType::class, array('label' => 'Reçu'));
        }
        $formMapper->end()
            ->with('Date', ['class' => 'col-md-4'])
                ->add('cotisationInfos.debut', DateTimeType::class, array(
                    'label' => 'Date de début',
                    'widget' => 'single_text',
                    // 'html5' => false,
                    'date_format' => IntlDateFormatter::SHORT,
                    // 'attr' => ['class' => 'js-datepicker'],
                ))
                ->add('cotisationInfos.fin', DateTimeType::class, array(
                    'label' => 'Date de fin',
                    'widget' => 'single_text',
                    // 'html5' => false,
                    'date_format' => IntlDateFormatter::SHORT,
                    // 'attr' => ['class' => 'js-datepicker'],
                ))
            ->end()
        ;
    }

    protected function configureRoutes(RouteCollection $collection)
    {
        $collection->remove('delete');
        if ($this->security->getUser() != null && !($this->security->getUser()->isGranted('ROLE_TRESORIER') || $this->security->getUser()->isGranted('ROLE_SUPER_ADMIN'))) {
            $collection->clearExcept('list');
        }
    }

    /**
    * {@inheritdoc}
    */
    protected function configureListFields(ListMapper $listMapper)
    {
        unset($this->listModes['mosaic']);
        $listMapper
            ->addIdentifier('cotisationInfos.annee', null, array(
                'label' => 'Année'
            ))
            ->addIdentifier('montant', null, array(
                'label' => 'Montant'
            ))
            ->addIdentifier('moyen', null, array(
                'label' => 'Moyen'
            ))
            ->addIdentifier('createdAt', null, array(
                'label' => 'Crée le'
            ))
            ->addIdentifier('operateur', null, array(
                'label' => 'Opérateur',
                // 'disabled' => true
            ))
        ;
    }
}
