<?php

namespace App\Entity;

use App\Entity\EntityTrait\EnablableEntityTrait;
use App\Enum\MoyenEnum;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * FLUX = TRANSFERT ou TRANSACTION ou COTISATIONS
 * @ORM\Entity(repositoryClass="App\Repository\FluxRepository")
 * @ORM\HasLifecycleCallbacks()
 * @ORM\InheritanceType("SINGLE_TABLE")
 * @ORM\DiscriminatorColumn(name="discr", type="string")
 * @ORM\DiscriminatorMap({"cotisation" = "Cotisation", "cotisation_adherent" = "CotisationAdherent", "cotisation_prestataire" = "CotisationPrestataire", "tro_adh_pre" = "TransactionAdherentPrestataire", "tro_adh_adh" = "TransactionAdherentAdherent", "tro_pre_adh" = "TransactionPrestataireAdherent", "tro_pre_pre" = "TransactionPrestatairePrestataire", "tre_cpt_adh" = "TransfertComptoirAdherent", "tre_cpt_grp" = "TransfertComptoirGroupe", "tre_cpt_pre" = "TransfertComptoirPrestataire", "tre_grp_cpt" = "TransfertGroupeComptoir", "tre_pre_cpt" = "TransfertPrestataireComptoir", "tre_pre_sie" = "TransfertPrestataireSiege", "tre_sie_grp" = "TransfertSiegeGroupe", "tre_grp_sie" = "TransfertGroupeSiege"})
 */
abstract class Flux
{
    use TimestampableEntity;

    /**
     * @var int
     *
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * @ORM\ManyToOne(targetEntity="User", inversedBy="flux", cascade={"all"})
     * @ORM\JoinColumn(name="user_id", referencedColumnName="id", nullable=true)
     * @Assert\NotBlank
     */
    protected $operateur;

    /**
     * Type de transfert / transaction : exemple : Prestataire à Adhérent
     * @var string
     *
     * @ORM\Column(name="type", type="string", length=200)
     * @Assert\NotBlank
     */
    protected $type;

    /**
     * Type de flux : transfert / transaction
     * @var string
     *
     * @ORM\Column(name="parenttype", type="string", length=20)
     * @Assert\NotBlank
     */
    protected $parenttype;

    /**
     * @var float
     *
     * @ORM\Column(name="montant", type="decimal", precision=7, scale=2)
     * @Assert\Type("float")
     * @Assert\GreaterThan(
     *     value = 0
     * )
     */
    protected $montant;

    /**
     * @var string
     *
     * @ORM\Column(name="moyen", type="string", length=100)
     * @Assert\NotBlank
     */
    private $moyen;

    /**
     * @var null|string
     *
     * @ORM\Column(name="reference", type="string", length=255, nullable=true)
     * @Assert\NotBlank
     */
    protected $reference;

    protected $expediteur = null;
    protected $destinataire = null;
    protected $cotisationInfos = null;

    abstract public function getParenttype();

    public function __construct()
    {
        $this->parenttype = $this->getParenttype();
        $this->type = $this->getType();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    /**
     * @param $parenttype
     * @return $this
     */
    public function setParenttype($parenttype)
    {
        $this->parenttype = $parenttype;
        return $this;
    }

    /**
     * @param User $destinataire
     * @return $this
     */
    public function setOperateur(User $operateur)
    {
        $this->operateur = $operateur;
        return $this;
    }

    /**
     * @return User operateur
     */
    public function getOperateur(): ?User
    {
        return $this->operateur;
    }

    /**
     * @param $destinataire
     * @return $this
     */
    public function setDestinataire($destinataire)
    {
        $this->destinataire = $destinataire;
        return $this;
    }

    /**
     * @return destinataire
     */
    public function getDestinataire()
    {
        return $this->destinataire;
    }

    /**
     * @param $expediteur
     * @return $this
     */
    public function setExpediteur($expediteur)
    {
        $this->expediteur = $expediteur;
        return $this;
    }

    /**
     * @return expediteur
     */
    public function getExpediteur()
    {
        return $this->expediteur;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @param string $type
     * @return Transaction
     */
    public function setType(string $type)
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return float
     */
    public function getMontant(): ?float
    {
        return $this->montant;
    }

    /**
     * @param float $montant
     * @return Transaction
     */
    public function setMontant(float $montant)
    {
        $this->montant = $montant;
        return $this;
    }

    /**
     * @return string
     */
    public function getReference(): ?string
    {
        return $this->reference;
    }

    /**
     * @param string $reference
     * @return Transaction
     */
    public function setReference(string $reference)
    {
        $this->reference = $reference;
        return $this;
    }

    /**
     * @return string
     */
    public function getMoyen(): ?string
    {
        return $this->moyen;
    }

    public function setMoyen($moyen)
    {
        if (!in_array($moyen, MoyenEnum::getAvailableTypes())) {
            throw new \InvalidArgumentException("Moyen de paiement invalide !");
        }
        $this->moyen = $moyen;

        return $this;
    }

    /**
     * @ORM\PostPersist
     * @param LifecycleEventArgs $event
     */
    public function postPersist(LifecycleEventArgs $event)
    {
        $flux = $event->getEntity();
        if (empty($flux->getExpediteur())) {
            throw new \Exception("[FLUX] Opération impossible ! Pas d'expéditeur !");
        }
        if ($flux->getMontant() <= 0) {
            throw new \Exception("[FLUX] Opération impossible ! Montant inférieur ou égal à zéro !");
        }
        if ($flux->getExpediteur() == $flux->getDestinataire()) {
            throw new \Exception("[FLUX] Opération impossible ! Expediteur et destinataire sont les mêmes !");
        }
        $compteExp = $flux->getExpediteur()->getCompte() - $flux->getMontant();
        if ($flux->getParenttype() != 'cotisation' || ($flux->getParenttype() == 'cotisation' && $flux->getMoyen() == MoyenEnum::MOYEN_MLC)) {
            if ($compteExp < 0) {
                throw new \Exception("[FLUX] Opération impossible ! Montant supérieur au solde de l'expéditeur !");
            } else {
                $em = $event->getEntityManager();
                $flux->getExpediteur()->setCompte($compteExp);

                $compteDest = $flux->getDestinataire()->getCompte() + $flux->getMontant();
                $flux->getDestinataire()->setCompte($compteDest);

                $em->persist($flux->getExpediteur());
                $em->persist($flux->getDestinataire());
                $em->flush();
            }
        }
    }

    public function toHtmlArray(): string
    {
        if (empty($this->getDestinataire()) || empty($this->getExpediteur()) || empty($this->getMontant())) {
            return "[FLUX] Visualisation impossible ! Destinataire / Expéditeur et/ou montant manquant(s) !";
        }
        $return = '<tr>';
        $return .= '<td>'.$this->getCreatedAt()->format('d/m/Y H:i').'</td>';
        $return .= '<td>'.ucwords($this->getParenttype()).'</td>';
        $return .= '<td>'.$this->getExpediteur().'</td>';
        $return .= '<td>'.$this->getDestinataire().'</td>';
        $return .= '<td>'.$this->getMontant().'&euro;</td>';
        $return .= '</tr>';
        return $return;
    }

    public function __toString(): string
    {
        if (empty($this->getDestinataire()) || empty($this->getExpediteur()) || empty($this->getMontant())) {
            return "[FLUX] Visualisation impossible ! Destinataire / Expéditeur et/ou montant manquant(s) !";
        }
        return $this->getCreatedAt()->format('d/m/Y H:i').' | '.ucwords($this->getParenttype()).' : '.$this->getDestinataire().' => '.$this->getExpediteur().' ; '.$this->getMontant().'&euro;';
    }
}
