<?php

namespace App\Controller;

use App\Events\MLCEvents;
use App\Events\FluxEvent;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Form;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Translation\TranslatorInterface;
use Payum\Core\Payum;
use Payum\Core\Request\GetHumanStatus;
use Payum\Core\Request\Notify;
use App\Entity\Payment;
use App\Entity\Siege;
use App\Entity\User;
use App\Entity\Adherent;
use App\Entity\Prestataire;
use App\Entity\AchatMonnaieAdherent;
use App\Entity\AchatMonnaiePrestataire;
use App\Entity\CotisationAdherent;
use App\Entity\CotisationPrestataire;

use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;

/**
 *
 * Types de transfert : (Les transferts dans la structure sont les flux de billets détenus par les opérateurs.)
 *
 *  - SIEGE             =>     GROUPES LOCAUX           (Transfert du siège au groupe)
 *  - GROUPE            =>     SIEGE                    (Transfert du groupe au siège)
 *  - GROUPES LOCAUX    =>     COMPTOIRS                (Transfert du groupe au comptoir)
 *  - COMPTOIRS         =>     GROUPES LOCAUX           (Transfert du comptoir au groupe)
 *  - COMPTOIRS         =>     ADHERENTS                (Diffusion de monnaie papier auprès des adhérents)
 *  - COMPTOIRS         =>     PRESTATAIRES             (Diffusion de monnaie papier auprès des prestataires)
 *  - PRESTATAIRES      =>     COMPTOIRS                (Reconversion)
 *
 * Types de transaction :
 *
 *   - PRESTATAIRES     =>    ADHERENTS         (Virement vers un adherent)
 *   - PRESTATAIRES     =>    PRESTATAIRES      (Virement entre prestataires)
 *   - ADHERENTS        =>    PRESTATAIRES      (Paiement numérique)
 *   - SIEGE            =>    ADHERENTS         (Achat de monnaie numérique par CB d'un adhérent)
 *   - SIEGE            =>    PRESTATAIRES      (Achat de monnaie numérique par CB d'un prestataire)
 *
 */
class FluxController extends AbstractController
{
    protected $em;
    protected $translator;
    protected $eventDispatcher;
    protected $session;
    protected $payum;

    public function __construct(Security $security, EntityManagerInterface $em, TranslatorInterface $translator, EventDispatcherInterface $eventDispatcher, SessionInterface $session, Payum $payum)
    {
        $this->security = $security;
        $this->em = $em;
        $this->translator = $translator;
        $this->eventDispatcher = $eventDispatcher;
        $this->session = $session;
        $this->payum = $payum;
    }

    protected function manageFluxForm(Request $request, Form $form, $compte, $success, $title)
    {
        if ($this->security->getUser() == null) {
            throw new \Exception("[FLUX] Opération impossible ! Utilisateur déconnecté !");
        }
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $data = $form->getData();
            $this->em->persist($data);
            $this->em->flush();
            $this->addFlash(
                'success',
                $success
            );
            $this->eventDispatcher->dispatch(
                MLCEvents::FLUX,
                new FluxEvent($data)
            );
            $referer = $request->headers->get('referer');
            if ($referer && !$request->isXmlHttpRequest()) {
                return $this->redirect($referer);
            } elseif (!$request->isXmlHttpRequest()) {
                return new Response('', Response::HTTP_BAD_REQUEST);
            }
        }

        return $this->render('flux/transaction.html.twig', [
            'form' => $form->createView(),
            'title' => $title
        ]);
    }

    /*
    * Crée une instance de Payment et redirige vers la page de paiement
    */
    public function preparePaymentAction(Form $form, $type)
    {
        $flux_data = $form->getData();

        // Enregistre les données du Flux en json, pour l'enregistrer une fois le paiement validé
        $serializer = $this->container->get('serializer');
        $data = $serializer->normalize($flux_data, null,
          [AbstractNormalizer::ATTRIBUTES => ['reference',
                                              'moyen',
                                              'montant',
                                              'role',
                                              'expediteur' => ['id'],
                                              'destinataire' => ['id'],
                                              'operateur' => ['id']]]);

        $jsondata = $serializer->serialize($data, 'json');

        // Prepare CB Payment
        $gatewayName = 'payzen';

        $storage = $this->payum->getStorage('App\Entity\Payment');

        $payment = $storage->create();
        $payment->setNumber(uniqid());
        $payment->setCurrencyCode('978');
        $payment->setTotalAmount($form->get('montant')->getData()*100); // 1.23 EUR
        $payment->setDescription($type);
        $payment->setClientId($this->getUser()->getId());
        $payment->setClientEmail($this->getUser()->getEmail());
        $payment->setFluxData($jsondata);

        $storage->update($payment);

        $captureToken = $this->payum->getTokenFactory()->createCaptureToken(
          $gatewayName,
          $payment,
          'payment_done' // the route to redirect after capture
        );

        return $this->redirect($captureToken->getTargetUrl());
    }

    /**
     * Fonction de traitement du paiement, à appeler :
     *  - automatiquement lorsqu'un événement se produit sur le site de Paiement
     *  - au retour sur le site par l'utilisateur
     *
     * @Route("/payment/done/", name="payment_done")
     */
    public function doneAction(Request $request)
    {
        try {
          $token = $this->payum->getHttpRequestVerifier()->verify($request);
        } catch (\Exception  $e) {
          // Token expiré : retour sur site après paiement

          // Get last payment
          $payment = $this->em->getRepository(Payment::class)->getUserLastPayment($this->getUser()->getId());
          if (!is_null($payment)) {

            if ($payment->getStatus() == GetHumanStatus::STATUS_CAPTURED || $payment->getStatus() == GetHumanStatus::STATUS_AUTHORIZED) {
                $type = $payment->getDescription();

                if (Payment::TYPE_ACHAT_MONNAIE_ADHERENT == $type || Payment::TYPE_ACHAT_MONNAIE_PRESTA == $type) {
                  $this->addFlash(
                    'success',
                    $this->translator->trans('Achat de monnaie locale bien effectué !')
                  );
                } else {
                  $this->addFlash(
                    'success',
                    $this->translator->trans('Cotisation bien reçue. Merci !')
                  );
                }

                // Update payment status
                $payment->setStatus($payment->getStatus().';done');
                $this->em->persist($payment);
                $this->em->flush();
            } else if ($payment->getStatus() == GetHumanStatus::STATUS_CANCELED ||
                        $payment->getStatus() == GetHumanStatus::STATUS_EXPIRED ||
                        $payment->getStatus() == GetHumanStatus::STATUS_FAILED) {

                $this->addFlash(
                  'error',
                  $this->translator->trans('La transaction a été annulée.')
                );

                $payment->setStatus($payment->getStatus().';done');
                $this->em->persist($payment);
                $this->em->flush();
            }
          }

          return $this->redirectToRoute('index');
        }

        $gateway = $this->payum->getGateway($token->getGatewayName());

        // Execute Notify action
        $gateway->execute(new Notify($token));

        // Execute 'done' action according to payment status
        $gateway->execute($status = new GetHumanStatus($token));

        // Get payment & update status
        $payment = $status->getFirstModel();
        $payment->setStatus($status->getValue());
        $this->em->persist($payment);
        $this->em->flush();

        // If payment succesful, persist serialized 'Flux' stored in payment
        if ($status->getValue() == GetHumanStatus::STATUS_CAPTURED || $status->getValue() == GetHumanStatus::STATUS_AUTHORIZED) {

            $serializer = $this->container->get('serializer');
            $flux_array = json_decode($payment->getFluxData(), true);

            $type = $payment->getDescription();
            if (Payment::TYPE_ACHAT_MONNAIE_ADHERENT == $type) {
                $flux = $serializer->deserialize(
                  $payment->getFluxData(),
                  AchatMonnaieAdherent::class,
                  'json',
                  ['disable_type_enforcement' => true]
                );

                $exp = $this->em->getRepository(Siege::class)->find($flux_array['expediteur']);
                $flux->setExpediteur($exp);

                $dest = $this->em->getRepository(Adherent::class)->find($flux_array['destinataire']);
                $flux->setDestinataire($dest);
            } else if (Payment::TYPE_ACHAT_MONNAIE_PRESTA == $type) {
                $flux = $serializer->deserialize(
                  $payment->getFluxData(),
                  AchatMonnaiePrestataire::class,
                  'json',
                  ['disable_type_enforcement' => true]
                );

                $exp = $this->em->getRepository(Siege::class)->find($flux_array['expediteur']);
                $flux->setExpediteur($exp);

                $dest = $this->em->getRepository(Prestataire::class)->find($flux_array['destinataire']);
                $flux->setDestinataire($dest);
            } else if (Payment::TYPE_COTISATION_ADHERENT == $type) {
                $flux = $serializer->deserialize(
                  $payment->getFluxData(),
                  CotisationAdherent::class,
                  'json',
                  ['disable_type_enforcement' => true]
                );

                $exp = $this->em->getRepository(Adherent::class)->find($flux_array['expediteur']);
                $flux->setExpediteur($exp);

                $dest = $this->em->getRepository(Prestataire::class)->find($flux_array['destinataire']);
                $flux->setDestinataire($dest);

                $flux->setRecu(true);
            } else if (Payment::TYPE_COTISATION_PRESTA == $type) {
                $flux = $serializer->deserialize(
                  $payment->getFluxData(),
                  CotisationPrestataire::class,
                  'json',
                  ['disable_type_enforcement' => true]
                );

                $exp = $this->em->getRepository(Prestataire::class)->find($flux_array['expediteur']);
                $flux->setExpediteur($exp);

                $dest = $this->em->getRepository(Prestataire::class)->find($flux_array['destinataire']);
                $flux->setDestinataire($dest);

                $flux->setRecu(true);
            } else {
                return new Response('', Response::HTTP_BAD_REQUEST);
            }

            $op = $this->em->getRepository(User::class)->find($flux_array['operateur']);
            $flux->setOperateur($op);

            $this->em->persist($flux);
            $this->em->flush();

            $this->eventDispatcher->dispatch(
                MLCEvents::FLUX,
                new FluxEvent($flux)
            );

            // Add Flash message here in case Gataway doesn't notify
            $this->addFlash(
              'success',
              $this->translator->trans('Achat de monnaie locale bien effectué !')
            );

            // Invalidate token
            $this->payum->getHttpRequestVerifier()->invalidate($token);

        } else if ($status->getValue() == GetHumanStatus::STATUS_CANCELED ||
                    $status->getValue() == GetHumanStatus::STATUS_EXPIRED ||
                    $status->getValue() == GetHumanStatus::STATUS_FAILED) {

            $this->addFlash(
              'error',
              $this->translator->trans('La transaction a été annulée.')
            );

            $this->payum->getHttpRequestVerifier()->invalidate($token);
        }

        return $this->redirectToRoute('index');
    }

}
