<?php

namespace App\Entity;

use App\Entity\EntityTrait\EnablableEntityTrait;
use App\Enum\MoyenEnum;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * FLUX = TRANSFERT ou TRANSACTION
 * @ORM\Entity
 * @ORM\InheritanceType("SINGLE_TABLE")
 * @ORM\DiscriminatorColumn(name="discr", type="string")
 * @ORM\DiscriminatorMap({"cotisation" = "Cotisation", "cotisation_adherent" = "CotisationAdherent", "cotisation_prestataire" = "CotisationPrestataire", "tro_adh_pre" = "TransactionAdherentPrestataire", "tro_pre_adh" = "TransactionPrestataireAdherent", "tro_pre_pre" = "TransactionPrestatairePrestataire", "tre_cpt_adh" = "TransfertComptoirAdherent", "tre_cpt_grp" = "TransfertComptoirGroupe", "tre_cpt_pre" = "TransfertComptoirPrestataire", "tre_grp_cpt" = "TransfertGroupeComptoir", "tre_pre_cpt" = "TransfertPrestataireComptoir", "tre_sie_grp" = "TransfertSiegeGroupe"})
 */
abstract class Flux
{
    use TimestampableEntity;

    /**
     * @var int
     *
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * @ORM\ManyToOne(targetEntity="User", inversedBy="flux", cascade={"all"})
     * @ORM\JoinColumn(name="user_id", referencedColumnName="id", nullable=true)
     */
    protected $operateur;

    /**
     * Type de transfert / transaction : exemple : Prestataire à Adhérent
     * @var string
     *
     * @ORM\Column(name="type", type="string", length=200)
     */
    protected $type;

    /**
     * Type de flux : transfert / transaction
     * @var string
     *
     * @ORM\Column(name="parenttype", type="string", length=20)
     */
    protected $parenttype;

    /**
     * @var float
     *
     * @ORM\Column(name="montant", type="decimal", precision=7, scale=2)
     */
    protected $montant;

    /**
     * @var string
     *
     * @ORM\Column(name="moyen", type="string", length=100)
     */
    private $moyen;

    /**
     * @var null|string
     *
     * @ORM\Column(name="reference", type="string", length=255, nullable=true)
     */
    protected $reference;

    protected $expediteur = null;
    protected $destinataire = null;
    protected $cotisationInfos = null;

    abstract public function getParenttype();

    public function __construct()
    {
        $this->parenttype = $this->getParenttype();
        $this->type = $this->getType();
    }

    public function getId(): int
    {
        return $this->id;
    }

    /**
     * @param $parenttype
     * @return $this
     */
    public function setParenttype($parenttype)
    {
        $this->parenttype = $parenttype;
        return $this;
    }

    /**
     * @param User $destinataire
     * @return $this
     */
    public function setOperateur(User $operateur)
    {
        $this->operateur = $operateur;
        return $this;
    }

    /**
     * @return User operateur
     */
    public function getOperateur(): User
    {
        return $this->operateur;
    }

    /**
     * @param $destinataire
     * @return $this
     */
    public function setDestinataire($destinataire)
    {
        $this->destinataire = $destinataire;
        return $this;
    }

    /**
     * @return destinataire
     */
    public function getDestinataire()
    {
        return $this->destinataire;
    }

    /**
     * @param $expediteur
     * @return $this
     */
    public function setExpediteur($expediteur)
    {
        $this->expediteur = $expediteur;
        return $this;
    }

    /**
     * @return expediteur
     */
    public function getExpediteur()
    {
        return $this->expediteur;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @param string $type
     * @return Transaction
     */
    public function setType(string $type)
    {
        $this->type = $type;
        return $this;
    }

    /**
     * @return float
     */
    public function getMontant(): ?float
    {
        return $this->montant;
    }

    /**
     * @param float $montant
     * @return Transaction
     */
    public function setMontant(float $montant)
    {
        $this->montant = $montant;
        return $this;
    }

    /**
     * @return string
     */
    public function getReference(): ?string
    {
        return $this->reference;
    }

    /**
     * @param string $reference
     * @return Transaction
     */
    public function setReference(string $reference)
    {
        $this->reference = $reference;
        return $this;
    }

    /**
     * @return string
     */
    public function getMoyen(): ?string
    {
        return $this->moyen;
    }

    public function setMoyen($moyen)
    {
        if (!in_array($moyen, MoyenEnum::getAvailableTypes())) {
            throw new \InvalidArgumentException("Moyen de paiement invalide !");
        }
        $this->moyen = $moyen;

        return $this;
    }
}
