<?php

namespace App\Entity;

use App\Entity\GlobalParameter;
use App\Entity\Prestataire;
use App\Enum\MoyenEnum;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;
use ApiPlatform\Core\Annotation\ApiResource;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * TRANSFERT
 *  - PRESTATAIRES      =>     SIEGE                (Reconversion)
 *
 * @ORM\Entity()
 */
class TransfertPrestataireSiege extends Transfert
{
    /**
     * @ORM\OneToOne(targetEntity="Prestataire")
     * @ORM\JoinColumn(name="prestataire_id", referencedColumnName="id")
     */
    protected $expediteur;

    /**
     * @ORM\OneToOne(targetEntity="Siege")
     * @ORM\JoinColumn(name="siege_id", referencedColumnName="id")
     */
    protected $destinataire;

    /**
     * @var boolean
     * @Assert\Type("bool")
     * @ORM\Column(type="boolean")
     * @Groups({"read", "write"})
     */
    protected $reconverti = false;

    public function __construct()
    {
        parent::__construct();
        $this->reconverti = false;
    }

    public function getReconverti(): bool
    {
        return $this->reconverti;
    }

    public function setReconverti(bool $reconverti)
    {
        $this->reconverti = $reconverti;
        return $this;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return 'reconversion';
    }

    public function operate($em)
    {
        $this->getExpediteur()->removeEcompte($this->getMontant());

        // On récupère le taux de reconversion (du formulaire s'il est précisé sinon par défaut dans la configuration globale)
        $taux = $this->getTauxreconversion();
        if (empty($taux)) {
            $taux = $em->getRepository(GlobalParameter::class)->val(GlobalParameter::RECONVERSION_PRESTATAIRE);
        }
        if ($taux > 0) {
            // On calcul les montants à virer au prestataire et la commission
            $montantAVirer = $this->getMontant() - ($this->getMontant()*($taux/100));
            $montantPreleve = $this->getMontant() - $montantAVirer;
            $this->setMontant($montantAVirer);
            $this->setTauxreconversion($taux);

            // On récupère le presta qui reçoit les commissions
            $mlcPrestataire = $em->getRepository(Prestataire::class)->findOneBy(array('mlc' => true));

            //Création d'un flux supplémentaire pour représenter la commission prise au prestataire et transféré à l'association qui gère la MLC
            $fluxCommission = new CommissionPrestataireMLC();
            $fluxCommission->setExpediteur($this->getExpediteur());
            $fluxCommission->setOperateur($this->getOperateur());
            $fluxCommission->setRole($this->getRole());
            $fluxCommission->setDestinataire($mlcPrestataire);
            $fluxCommission->setMontant($montantPreleve);
            $fluxCommission->setReference('Commission de reconversion');
            $fluxCommission->setMoyen(MoyenEnum::MOYEN_TRANSFERT);
            $fluxCommission->setTauxreconversion($taux);

            $mlcPrestataire->addEcompte($montantPreleve);

            return [$this->getExpediteur(), $this->getDestinataire(), $fluxCommission, $mlcPrestataire];
        }

        return [$this->getExpediteur(), $this->getDestinataire()];
    }

    public function getUsersToNotify()
    {
        return $this->getExpediteur()->getUsers()->toArray();
    }
}
