<?php

namespace App\Admin;

use App\Entity\User;
use App\Entity\Flux;
use Sonata\AdminBundle\Admin\AbstractAdmin;
use Sonata\AdminBundle\Datagrid\DatagridMapper;
use Sonata\AdminBundle\Datagrid\ListMapper;
use Sonata\AdminBundle\Form\FormMapper;
use Sonata\AdminBundle\Form\Type\Filter\ChoiceType;
use Sonata\AdminBundle\Route\RouteCollection;
use Sonata\AdminBundle\Show\ShowMapper;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType as SChoiceType;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * Administration des Flux
 *
 * KOHINOS : Outil de gestion de Monnaie Locale Complémentaire
 * @author Julien Jorry <julien.jorry@gmail.com>
 */
class FluxAdmin extends AbstractAdmin
{
    protected $translator;
    protected $security;
    protected $datagridValues = [
        '_sort_order' => 'DESC',
        '_sort_by' => 'createdAt',
    ];

    public function setTranslator(TranslatorInterface $translator)
    {
        $this->translator = $translator;
    }

    public function setSecurity(Security $security)
    {
        $this->security = $security;
    }

    /**
    * {@inheritdoc}
    */
    public function createQuery($context = 'list')
    {
        $query = $this->getModelManager()->createQuery(Flux::class);
        foreach ($this->extensions as $extension) {
            $extension->configureQuery($this, $query, $context);
        }
        return $query;
    }

    /**
    * {@inheritdoc}
    */
    protected function configureRoutes(RouteCollection $collection)
    {
        $collection->clearExcept(array('list', 'export'));
    }

    /**
    * {@inheritdoc}
    */
    protected function configureDatagridFilters(DatagridMapper $datagridMapper): void
    {
        $datagridMapper
            ->add('transfert_or_transaction', 'doctrine_orm_callback', array(
                'label' => "Type",
                'callback' => function ($queryBuilder, $alias, $field, $value) {
                    if (!$value['value']) {
                        return;
                    }
                    $queryBuilder
                        ->where($alias.".parenttype = :type")
                        ->setParameter('type', $value['value']);
                    return true;
                },
                'advanced_filter' => false,
                'show_filter' => true,
                'field_type' => SChoiceType::class,
                'field_options' => array(
                    'choices' => array('Transactions' => 'transaction', 'Transferts' => 'transfert', 'Cotisations' => 'cotisation', 'Achat de monnaie' => 'achat'),
                    'placeholder' => 'Indifférent',
                    'expanded' => true,
                    'multiple' => false
                )
            ))
            ->add('type', null, array(
                'label' => 'Type plus précis',
                'advanced_filter' => false,
                'show_filter' => true
            ))
            ->add('operateur', null, array(
                'label' => 'Operateur',
                'advanced_filter' => false,
                'show_filter' => true
            ))
        ;
    }

    public function getTemplate($name)
    {
        if ($name == 'list') {
            return 'block/base_list_with_total.html.twig';
        }

        return parent::getTemplate($name);
    }

    public function getTotalLabel()
    {
        return $this->translator->trans('Total des flux :');
    }

    public function getTotal()
    {
        $datagrid = $this->getDatagrid();
        $datagrid->buildPager();

        $query = clone $datagrid->getQuery();
        $query
            ->select('SUM( ' . $query->getRootAlias() . '.montant) as total')
            ->setFirstResult(null)
            ->setMaxResults(null);

        $result = $query->execute(array(), \Doctrine\ORM\Query::HYDRATE_SINGLE_SCALAR);

        return $result;
    }

    /**
    * {@inheritdoc}
    */
    protected function configureListFields(ListMapper $listMapper)
    {
        unset($this->listModes['mosaic']);
        $listMapper
            ->add('createdAt', 'datetime', array('label' => 'Date'))
            ->add('type', null, array('label' => 'Type'))
            ->add('montant', null, array('label' => 'Montant'))
            ->add('expediteur', null, array('label' => 'Expediteur'))
            ->add('destinataire', null, array('label' => 'Destinataire'))
            ->add('operateur', null, array('label' => 'Operateur'))
            ->addIdentifier('reference', null, array('label' => 'Reference'))
            // @TODO : ajouter le verify uniquement si l'on souhaite (param url)=> sinon c'est beaucoup trop long...
            // ->addIdentifier('verify', null, array('label' => 'Vérifié'))
        ;
    }
}
