<?php

/*
 * kohinos_cooperatic
 * Copyright (C) 2019-2020  ADML63
 * Copyright (C) 2020- Cooperatic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
 
namespace App\Entity;

use App\Entity\ContactComptoir;
use App\Entity\EntityTrait\EnablableEntityTrait;
use App\Entity\EntityTrait\GeolocEntityTrait;
use App\Entity\EntityTrait\HasCompteEntity;
use App\Entity\EntityTrait\NameSlugContentEntityTrait;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Symfony\Component\Validator\Constraints as Assert;
use ApiPlatform\Core\Annotation\ApiResource;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * @ApiResource(
 *     attributes={"security"="is_granted('ROLE_ADMIN_COMPTOIR_GERER_VIEW') or is_granted('ROLE_API')"},
 *     collectionOperations={
 *         "get"={"security"="is_granted('ROLE_ADMIN_COMPTOIR_GERER_LIST') or is_granted('ROLE_API')"},
 *         "post"={"security"="is_granted('ROLE_ADMIN_COMPTOIR_GERER_EDIT')"}
 *     },
 *     itemOperations={
 *         "get"={"security"="is_granted('ROLE_ADMIN_COMPTOIR_GERER_VIEW') or is_granted('ROLE_API')"},
 *         "put"={"security"="is_granted('ROLE_ADMIN_COMPTOIR_GERER_EDIT')"},
 *     },
 *     normalizationContext={"groups"={"read"}},
 *     denormalizationContext={"groups"={"write"}}
 * )
 * @ORM\Entity(repositoryClass="App\Repository\ComptoirRepository")
 * @ORM\HasLifecycleCallbacks()
 * @ORM\Table(name="comptoir")
 */
class Comptoir
{
    use NameSlugContentEntityTrait,
        TimestampableEntity,
        EnablableEntityTrait,
        GeolocEntityTrait,
        HasCompteEntity;

    /**
     * @var int
     *
     * @ORM\Id
     * @ORM\Column(name="id", type="integer")
     * @ORM\GeneratedValue
     */
    protected $id;

    /**
    * @var \Application\Sonata\MediaBundle\Entity\Media
    * @ORM\ManyToOne(targetEntity="App\Application\Sonata\MediaBundle\Entity\Media", cascade={"persist"}, fetch="LAZY")
    * @ORM\JoinColumn(name="media_id", referencedColumnName="id")
    */
    protected $media;

    /**
     * @var Groupe $comptoirGroup
     *
     * @ORM\ManyToOne(targetEntity="Groupe", inversedBy="comptoirs")
     * @ORM\OrderBy({"name": "ASC"})
     */
    private $groupe;

    /**
    * @var ArrayCollection|User[]
    *
    * @ORM\ManyToMany(targetEntity="User", inversedBy="comptoirsgeres", fetch="EXTRA_LAZY")
    */
    private $gestionnaires;

    /**
     * @var ArrayCollection|ContactComptoir[]
     * @ORM\OneToMany(targetEntity="ContactComptoir", cascade={"persist"}, mappedBy="comptoir")
     * @Groups({"read", "write"})
     */
    private $contacts;

    /**
     * @var ArrayCollection|User[]
     *
     * @ORM\ManyToMany(targetEntity="User", inversedBy="caissiercomptoirs", cascade={"persist"})
     * @ORM\JoinTable(name="user_comptoir_caissier",
     *      joinColumns={@ORM\JoinColumn(name="comptoir_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="user_id", referencedColumnName="id")}
     * )
     * @Groups({"read", "write"})
     */
    private $caissiers;

    public function __construct()
    {
        $this->gestionnaires = new ArrayCollection();
        $this->contacts = new ArrayCollection();
        $this->caissiers = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }

    /**
    * Get media
    * @return
    */
    public function getMedia()
    {
        return $this->media;
    }

    /**
    * Set media
    * @return $this
    */
    public function setMedia($media): self
    {
        $this->media = $media;
        return $this;
    }

    /**
     * @param null|Groupe $groupes
     * @return $this
     */
    public function setGroupe(?Groupe $groupe): self
    {
        $this->groupe = $groupe;
        return $this;
    }

    /**
     * @return null|Groupe
     */
    public function getGroupe(): ?Groupe
    {
        return $this->groupe;
    }

    /**
     * @return User[]|ArrayCollection
     */
    public function getGestionnaires()
    {
        return $this->gestionnaires;
    }

    /**
     * @param User[]|ArrayCollection
     * @return $this
     */
    public function setGestionnaires($gestionnaires): self
    {
        $this->gestionnaires = $gestionnaires;
        return $this;
    }

    /**
     * @param User $gestionnaire
     * @return $this
     */
    public function addGestionnaire(User $gestionnaire): self
    {
        if (!$this->gestionnaires->contains($gestionnaire)) {
            $this->gestionnaires[] = $gestionnaire;
        }
        return $this;
    }

    /**
     * @param User $gestionnaire
     * @return $this
     */
    public function removeGestionnaire(User $gestionnaire): self
    {
        if ($this->gestionnaires->contains($gestionnaire)) {
            $this->gestionnaires->removeElement($gestionnaire);
        }
        return $this;
    }

    /**
     * getcontacts
     * @return ArrayCollection contact
     */
    public function getContacts()
    {
        return $this->contacts;
    }

    /**
     * setContacts
     * @param [type] $contacts [description]
     */
    public function setContacts($contacts): self
    {
        $this->contacts = $contacts;
        return $this;
    }

    /**
     * @param ContactComptoir $contact
     * @return $this
     */
    public function addContact(ContactComptoir $contact): self
    {
        if (is_null($this->contacts)) {
            $this->contacts = new ArrayCollection();
        }
        if (!$this->contacts->contains($contact)) {
            $this->contacts[] = $contact;
            $contact->setComptoir($this);
        }
        return $this;
    }

    /**
     * @param ContactComptoir $contact
     * @return $this
     */
    public function removeContact(ContactComptoir $contact): self
    {
        if ($this->contacts->contains($contact)) {
            $this->contacts->removeElement($contact);
            $contact->setComptoir(null);
        }
        return $this;
    }


    public function __toString(): string
    {
        return (!empty($this->name)?$this->name:'Comptoir');
    }

    /**
     * @return Collection|User[]
     */
    public function getCaissiers(): Collection
    {
        return $this->caissiers;
    }

    public function getCaissiersString()
    {
        return join(' - ', array_map(function ($user) {
            return $user->getName() . ':' . $user->getEmail();
        }, $this->caissiers->getValues()));
    }

    public function addCaissier(User $caissier): self
    {
        if (!$this->caissiers->contains($caissier)) {
            $this->caissiers[] = $caissier;
        }

        return $this;
    }

    /**
     * @param ArrayCollection[]|User $caissiers
     *
     * @return Comptoir
     */
    public function setCaissiers($caissiers): self
    {
        $this->caissiers = $caissiers;

        return $this;
    }

    public function removeCaissier(User $caissier): self
    {
        if ($this->caissiers->contains($caissier)) {
            $this->caissiers->removeElement($caissier);
        }

        return $this;
    }
}
