<?php

namespace App\Entity;

use App\Entity\Adherent;
use App\Entity\Document;
use App\Entity\EmailToken;
use App\Entity\Faq;
use App\Entity\Flux;
use App\Entity\Prestataire;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Sonata\UserBundle\Entity\BaseUser as BaseUser;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity(repositoryClass="App\Repository\UserRepository")
 * @ORM\Table(name="user")
 * @UniqueEntity(
 *     fields={"firstname", "lastname", "email"},
 *     errorPath="email",
 *     message="Ce nom avec cet email est déjà utilisé !"
 * )
 */
class User extends BaseUser
{
    /**
     * @ORM\Id
     * @ORM\Column(name="id", type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * @var null|string
     *
     * @ORM\Column(name="etat", type="string", length=10, nullable=true)
     */
    protected $etat;

    /**
     * @ORM\Column(type="string", length=15, nullable=true)
     */
    protected $mobile;

    /**
     * @ORM\OneToOne(targetEntity="Adherent", inversedBy="user", cascade={"all"})
     * @ORM\JoinColumn(name="adherent_id", referencedColumnName="id", nullable=true)
     */
    protected $adherent;

    /**
     * @ORM\OneToOne(targetEntity="Prestataire", inversedBy="user", cascade={"all"})
     * @ORM\JoinColumn(name="prestataire_id", referencedColumnName="id", nullable=true)
     */
    protected $prestataire;

    /**
     * @var ArrayCollection|Cotisation[]
     *
     * @ORM\OneToMany(targetEntity="Cotisation", mappedBy="user", cascade={"persist"})
     */
    private $cotisations;

    /**
     * @var ArrayCollection|Document[]
     *
     * @ORM\OneToMany(targetEntity="Document", mappedBy="user", cascade={"persist"})
     */
    private $documents;

    /**
     * @var ArrayCollection|Faq[]
     *
     * @ORM\OneToMany(targetEntity="Faq", mappedBy="user", cascade={"persist"})
     */
    private $faqs;

    /**
     * @ORM\OneToMany(targetEntity="Flux", mappedBy="operateur")
     */
    protected $flux;

    /**
     * @ORM\OneToMany(targetEntity="EmailToken", mappedBy="user", cascade={"persist", "remove"})
     * @ORM\OrderBy({"expiredAt" = "DESC"})
     */
    private $emailTokens;

    public function __construct()
    {
        parent::__construct();
        $this->cotisations = new ArrayCollection();
        $this->flux = new ArrayCollection();
        $this->emailTokens = new ArrayCollection();
    }

    public function getCommonName(): ?string
    {
        return $this->getFirstname()." ".$this->getLastname();
    }

    /**
     * @return null|string
     */
    public function getEtat(): ?string
    {
        return $this->etat;
    }

    /**
     * @param null|string $etat
     * @return Prestataire
     */
    public function setEtat(?string $etat)
    {
        $this->etat = $etat;
        return $this;
    }

    /**
     * Get mobile number
     * @return [type] [description]
     */
    public function getMobile(): ?string
    {
        return $this->mobile;
    }

    /**
     * Set mobile number
     * @param string $mobile [description]
     */
    public function setMobile(?string $mobile): self
    {
        $this->mobile = $mobile;

        return $this;
    }


    public function isGranted($role)
    {
        return in_array($role, $this->getRoles());
    }

    public function isAdmin()
    {
        if ($this->isSuperAdmin()) {
            return true;
        }
        $isAdmin = false;
        foreach ($this->getRoles() as $role) {
            if ((is_object($role) && $role->getRole() == 'ROLE_ADMIN') || $role == 'ROLE_ADMIN') {
                $isAdmin = true;
                break;
            }
        }
        return $isAdmin;
    }

    public function isSuperAdmin()
    {
        return $this->getSuperAdmin();
    }

    public function getSuperAdmin()
    {
        $isSuperAdmin = false;
        foreach ($this->getRoles() as $role) {
            if ((is_object($role) && $role->getRole() == 'ROLE_SUPER_ADMIN') || $role == 'ROLE_SUPER_ADMIN') {
                $isSuperAdmin = true;
                break;
            }
        }
        return $isSuperAdmin;
    }

    /**
    * Get adherent
    * @return
    */
    public function getAdherent()
    {
        return $this->adherent;
    }

    /**
    * Set adherent
    * @return $this
    */
    public function setAdherent($adherent): self
    {
        $this->adherent = $adherent;
        return $this;
    }

    /**
    * Get prestataire
    * @return
    */
    public function getPrestataire()
    {
        return $this->prestataire;
    }

    /**
    * Set prestataire
    * @return $this
    */
    public function setPrestataire($prestataire): self
    {
        $this->prestataire = $prestataire;
        return $this;
    }

    /**
     * @return Cotisation[]|ArrayCollection
     */
    public function getCotisations()
    {
        return $this->cotisations;
    }

    /**
     * @param Cotisation $cotisation
     * @return $this
     */
    public function addCotisation(Cotisation $cotisation)
    {
        if (!$this->cotisations->contains($cotisation)) {
            $this->cotisations[] = $cotisation;
            $cotisation->setUser($this);
        }
        return $this;
    }

    /**
     * @param Cotisation $cotisation
     * @return $this
     */
    public function removeCotisation(Cotisation $cotisation)
    {
        throw new \LogicException('User::removeCotisation : This code should not be reached!');
        // if ($this->cotisations->contains($cotisation)) {
        //     $this->cotisations->removeElement($cotisation);
        // }
        return $this;
    }

    /**
     * @return Flux[]|ArrayCollection
     */
    public function getFlux()
    {
        return $this->flux;
    }

    /**
     * @param Flux $flux
     * @return $this
     */
    public function addFlux(Flux $flux)
    {
        if (!$this->flux->contains($flux)) {
            $this->flux[] = $flux;
            $flux->setUser($this);
        }
        return $this;
    }

    /**
     * @param Flux $flux
     * @return $this
     */
    public function removeFlux(Flux $flux)
    {
        throw new \LogicException('User::removeFlux : This code should not be reached!');
        // if ($this->flux->contains($flux)) {
        //     $this->flux->removeElement($flux);
        // }
        return $this;
    }

    public function createEmailToken()
    {
        $token = new EmailToken();
        $token->setUser($this);
        $this->emailTokens->add($token);

        return $token;
    }

    public function getFirstValidEmailToken()
    {
        foreach ($this->getEmailTokens() as $emailToken) {
            if ($emailToken->isValid()) {
                return $emailToken;
            }
        }

        return null;
    }

    public function getEmailTokens()
    {
        return $this->emailTokens;
    }

    public function getEmailToken($token)
    {
        foreach ($this->emailTokens as $emailToken) {
            if ($emailToken->getToken()==$token) {
                return $emailToken;
            }
        }
    }
}
