<?php

namespace App\Admin;

use App\Admin\UserAdmin;
use App\Entity\Cotisation;
use App\Entity\Geoloc;
use App\Entity\Prestataire;
use App\Entity\User;
use App\Entity\Usergroup;
use App\Enum\MoyenEnum;
use Doctrine\ORM\Query;
use FOS\UserBundle\Model\UserManagerInterface;
use Knp\Menu\ItemInterface as MenuItemInterface;
use Sonata\AdminBundle\Admin\AbstractAdmin;
use Sonata\AdminBundle\Admin\AdminInterface;
use Sonata\AdminBundle\Datagrid\DatagridMapper;
use Sonata\AdminBundle\Datagrid\ListMapper;
use Sonata\AdminBundle\Form\FormMapper;
use Sonata\AdminBundle\Route\RouteCollection;
use Sonata\AdminBundle\Show\ShowMapper;
use Sonata\MediaBundle\Form\Type\MediaType;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

class PrestataireAdmin extends AbstractAdmin
{
    protected $baseRouteName = 'prestataire';
    protected $baseRoutePattern = 'prestataire';

    public function configure()
    {
        parent::configure();
    }

    protected function configureSideMenu(MenuItemInterface $menu, $action, AdminInterface $childAdmin = null)
    {
        if (!$childAdmin && !in_array($action, ['edit', 'show'])) {
            return;
        }

        $admin = $this->isChild() ? $this->getParent() : $this;
        $id = $admin->getRequest()->get('id');
        $user = $this->getConfigurationPool()->getContainer()->get('doctrine')->getRepository(User::class)->findOneBy(array('prestataire' => $id));

        if ($this->isGranted('EDIT') && $user != null) {
            $menu->addChild('Edit User', [
                'uri' => $this->getConfigurationPool()->getContainer()->get('router')->generate('admin_app_user_edit', ['id' => $user->getId()], UrlGeneratorInterface::ABSOLUTE_URL)
            ]);
        }
    }

    /**
    * {@inheritdoc}
    */
    protected function configureFormFields(FormMapper $formMapper): void
    {
        // Initialize prestataire
        $presta = $this->getSubject();
        $user = $this->userManager->createUser();
        $groupe = $this->getConfigurationPool()->getContainer()->get('doctrine')->getRepository(Usergroup::class)->findOneByName('Prestataire');
        $user->setEnabled(true);
        $user->addGroup($groupe);
        $user->addRole('ROLE_PRESTATAIRE');
        // $presta = new Prestataire();
        $presta->setGeoloc(new Geoloc());
        $now = new \DateTime();
        $cotisation = new Cotisation();
        $cotisation->setDebut($now);
        $cotisation->setFin(new \DateTime('+ 1 year'));
        $user->addCotisation($cotisation);
        $user->setPrestataire($presta);
        $presta->setUser($user);

        // get the current Image instance
        $imageHelp = null;
        if (!empty($user->getPrestataire()) && !empty($user->getPrestataire()->getMedia())) {
            $image = $user->getPrestataire()->getMedia();
            if ($image && ($webPath = $image->getWebPath())) {
                // get the container so the full path to the image can be set
                $container = $this->getConfigurationPool()->getContainer();
                $fullPath = $container->get('request_stack')->getCurrentRequest()->getBasePath().'/'.$webPath;
                // add a 'help' option containing the preview's img tag
                $imageHelp = '<img src="'.$fullPath.'" class="admin-preview" />';
            }
        }

        $formMapper
            ->tab('Prestataire')
                ->with('Identité', ['class' => 'col-md-7'])
                    ->add('user.firstname', TextType::class, array(
                        'label' => 'Prénom :',
                        'required' => true
                    ))
                    ->add('user.lastname', TextType::class, array(
                        'label' => 'Nom :',
                        'required' => true
                    ))
                    ->add('user.phone', TextType::class, array(
                        'label' => 'Téléphone :',
                        'required' => true
                    ))
                    ->add('user.email', TextType::class, array(
                        'label' => 'Email :',
                        'required' => true
                    ))
                ->end()
                ->with('Cotisation', ['class' => 'col-md-5'])
                    ->add('user.cotisations.first.annee', TextType::class, array('label' => 'Année', 'data' => $now->format('Y')))
                    ->add('user.cotisations.first.montant', TextType::class, array('label' => 'Montant'))
                    ->add('user.cotisations.first.moyen', ChoiceType::class, array(
                        'required' => true,
                        'choices' => MoyenEnum::getAvailableTypes(),
                        // 'choices_as_values' => true,
                        'choice_label' => function ($choice) {
                            return MoyenEnum::getTypeName($choice);
                        },
                    ))
                    ->add('user.cotisations.first.recu', CheckboxType::class, array('label' => 'Reçu'))
                ->end()
                ->with('Société', ['class' => 'col-md-7'])
                    ->add('raison', TextType::class, array(
                        'label' => 'Raison :',
                        'required' => true
                    ))
                    ->add('statut', TextType::class, array(
                        'label' => 'Statut :',
                        'required' => false
                    ))
                    ->add('siret', TextType::class, array(
                        'label' => 'SIRET :',
                        'required' => true
                    ))
                    ->add('iban', TextType::class, array(
                        'label' => 'IBAN :',
                        'required' => true
                    ))
                    ->add('responsable', TextType::class, array(
                        'label' => 'Responsable :',
                        'required' => false
                    ))
                    ->add('metier', TextType::class, array(
                        'label' => 'Métier responsable :',
                        'required' => true
                    ))
                ->end()
                ->with('Addresse', ['class' => 'col-md-5'])
                    ->add('geoloc.adresse', TextType::class, array(
                        'label' => 'Addresse :',
                        'required' => false
                    ))
                    ->add('geoloc.cpostal', TextType::class, array(
                        'label' => 'Code postal :',
                        'required' => false
                    ))
                    ->add('geoloc.ville', TextType::class, array(
                        'label' => 'Ville :',
                        'required' => false
                    ))
                    // ->add('geoloc.lat', TextType::class, array(
                    //     'label' => 'Latitude :',
                    //     'required' => false
                    // ))
                    // ->add('geoloc.lon', TextType::class, array(
                    //     'label' => 'Longitude :',
                    //     'required' => false
                    // ))
                ->end()
                ->with('Image', ['class' => 'col-md-5'])
                    ->add('media', MediaType::class, array(
                        'provider' => 'sonata.media.provider.image',
                        'context' => 'prestataire',
                        'help' => $imageHelp,
                        'required' => false
                    ))
                ->end()
            ->end()
        ;
        parent::configureFormFields($formMapper);
    }

    public function preUpdate($prestataire)
    {
        $this->prePersist($prestataire);
    }

    public function prePersist($prestataire)
    {
        $em = $this->getConfigurationPool()->getContainer()->get('doctrine')->getManager();

        if (empty($prestataire->getUser()->getUsername())) {
            $prestataire->getUser()->setUsername($prestataire->getUser()->getEmail());
        }
        if (empty($prestataire->getUser()->getPassword())) {
            // $tokenGenerator = $this->getConfigurationPool()->getContainer()->get('fos_user.util.token_generator');
            // $password = substr($tokenGenerator->generateToken(), 0, 12);
            $bytes = random_bytes(64);
            $password = rtrim(strtr(base64_encode($bytes), '+/', '-_'), '=');
            $prestataire->getUser()->setPassword($password);
            // send email to user
        }
        $this->userManager->updateUser($prestataire->getUser());
        $prestataire->getUser()->createEmailToken();
        $em->persist($prestataire->getUser());
        $em->persist($prestataire);
        $em->flush();
        // TODO : envoyer un mail au nouvel utilisateur avec l'emailtoken via le dispatch d'un event
        // $this->eventDispatcher->dispatch(FOSUserEvents::REGISTRATION_COMPLETED, new FilterUserResponseEvent($adherent->getUser(), $this->getRequest(), $response));
    }

    /**
    * {@inheritdoc}
    */
    protected function configureDatagridFilters(DatagridMapper $datagridMapper): void
    {
        parent::configureDatagridFilters($datagridMapper);
        $datagridMapper
            ->add('raison')
            ->add('statut')
        ;
    }

    /**
     * @param UserManagerInterface $userManager
     */
    public function setUserManager(UserManagerInterface $userManager): void
    {
        $this->userManager = $userManager;
    }

    /**
     * @return UserManagerInterface
     */
    public function getUserManager()
    {
        return $this->userManager;
    }

    protected function configureListFields(ListMapper $listMapper): void
    {
        unset($this->listModes['mosaic']);
        $listMapper
            ->addIdentifier('user.username')
            ->addIdentifier('user.email')
            ->addIdentifier('raison')
            ->addIdentifier('user.enabled', null, array('label' => 'Activé', 'datatype' => 'App.User', 'template' => '@SonataAdmin/Boolean/editable_boolean.html.twig'))
            ->addIdentifier('user.createdAt')
        ;

        // if ($this->isGranted('ROLE_ALLOWED_TO_SWITCH')) {
        //     $listMapper
        //         ->addIdentifier('user.impersonating', 'string', ['template' => '@SonataUser/Admin/Field/impersonating.html.twig'])
        //     ;
        // }
    }

    protected function configureRoutes(RouteCollection $collection)
    {
        $collection->remove('delete');
    }

    /**
    * {@inheritdoc}
    */
    public function createQuery($context = 'list')
    {
        $query = parent::createQuery($context);
        // $query->setHint(Query::HINT_FORCE_PARTIAL_LOAD, true);
        $query->innerJoin($query->getRootAliases()[0] .'.user', 'u')
                ->addSelect('u');
        return $query;
    }
}
