<?php

declare(strict_types=1);

/*
 * This file is part of the Sonata Project package.
 *
 * (c) Thomas Rabaix <thomas.rabaix@sonata-project.org>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace App\Application\Sonata\UserBundle\Admin;

use App\Entity\User;
use FOS\UserBundle\Model\UserManagerInterface;
use Sonata\AdminBundle\Admin\AbstractAdmin;
use Sonata\AdminBundle\Datagrid\DatagridMapper;
use Sonata\AdminBundle\Datagrid\ListMapper;
use Sonata\AdminBundle\Form\FormMapper;
use Sonata\AdminBundle\Form\Type\ModelType;
use Sonata\AdminBundle\Route\RouteCollection;
use Sonata\AdminBundle\Show\ShowMapper;
use Sonata\CoreBundle\Form\Type\DatePickerType;
use Sonata\UserBundle\Admin\Model\UserAdmin as BaseUserAdmin;
use Sonata\UserBundle\Form\Type\SecurityRolesType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\LocaleType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\TimezoneType;
use Symfony\Component\Form\Extension\Core\Type\UrlType;
use Symfony\Component\Form\FormTypeInterface;

class UserAdmin extends BaseUserAdmin
{
    public function configure()
    {
        parent::configure();
        $this->setTemplate('edit', '@SonataAdmin/base_edit.html.twig');
    }

    protected function configureRoutes(RouteCollection $collection)
    {
        if ($this->isChild()) {
            $collection->remove('delete');
            return;
        }

        // This is the route configuration as a parent
        $collection->remove('delete');
    }
    /**
     * {@inheritdoc}
     */
    protected function configureListFields(ListMapper $listMapper): void
    {
        $listMapper
            ->addIdentifier('username')
            ->add('email')
            ->add('groups')
            ->addIdentifier('enabled', null, array('label' => 'Activé', 'datatype' => 'App.User', 'template' => '@SonataAdmin/Boolean/editable_boolean.html.twig'))
            ->add('createdAt')
        ;

        if ($this->isGranted('ROLE_ALLOWED_TO_SWITCH')) {
            $listMapper
                ->add('impersonating', 'string', ['template' => '@SonataUser/Admin/Field/impersonating.html.twig'])
            ;
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function configureDatagridFilters(DatagridMapper $filterMapper): void
    {
        $filterMapper
            ->add('id')
            ->add('username')
            ->add('email')
            ->add('groups')
        ;
    }

    /**
     * {@inheritdoc}
     */
    protected function configureShowFields(ShowMapper $showMapper): void
    {
        $showMapper
            ->with('General')
                ->add('username')
                ->add('email')
            ->end()
            ->with('Groups')
                ->add('groups')
            ->end()
            ->with('Profile')
                // ->add('dateOfBirth')
                ->add('firstname')
                ->add('lastname')
                // ->add('website')
                // ->add('biography')
                // ->add('gender')
                // ->add('locale')
                // ->add('timezone')
                ->add('phone')
            ->end()
            // ->with('Social')
            //     ->add('facebookUid')
            //     ->add('facebookName')
            //     ->add('twitterUid')
            //     ->add('twitterName')
            //     ->add('gplusUid')
            //     ->add('gplusName')
            // ->end()
            // ->with('Security')
            //     ->add('token')
            //     ->add('twoStepVerificationCode')
            // ->end()
        ;
    }

    /**
     * {@inheritdoc}
     */
    protected function configureFormFields(FormMapper $formMapper): void
    {
        $subject = $this->getSubject();
        // define group zoning
        $formMapper
            ->tab('User')
                ->with('Profile', ['class' => 'col-md-6'])->end()
                ->with('General', ['class' => 'col-md-6'])->end()
                // ->with('Social', ['class' => 'col-md-6'])->end()
            ->end();
        if (!($subject->isGranted('ROLE_SUPER_ADMIN') || $subject->isGranted('ROLE_ADMIN_SIEGE'))) {
            $formMapper
                ->tab('Security')
                ->with('Groups', ['class' => 'col-md-8'])->end()
                ->with('Status', ['class' => 'col-md-4'])->end()
                // ->with('Keys', ['class' => 'col-md-4'])->end()
                // ->with('Roles', ['class' => 'col-md-12'])->end()
                ->end()
            ;
        }

        $now = new \DateTime();

        $genderOptions = [
            'choices' => call_user_func([$this->getUserManager()->getClass(), 'getGenderList']),
            'required' => true,
            'translation_domain' => $this->getTranslationDomain(),
        ];

        // w: Remove this when dropping support for SF 2.8
        if (method_exists(FormTypeInterface::class, 'setDefaultOptions')) {
            $genderOptions['choices_as_values'] = true;
        }

        // $formMapper->removeGroup('Social');
        $formMapper
            ->tab('User')
                ->with('General')
                    ->add('username')
                    ->add('email');
        if ($this->isGranted('ROLE_SUPER_ADMIN')) {
            $formMapper
                ->add('plainPassword', TextType::class, [
                    'required' => (!$this->getSubject() || null === $this->getSubject()->getId())
                ])
            ;
        }
        $formMapper->end()
                ->with('Profile')
                    // ->add('dateOfBirth', DatePickerType::class, [
                    //     'years' => range(1900, $now->format('Y')),
                    //     'dp_min_date' => '1-1-1900',
                    //     'dp_max_date' => $now->format('c'),
                    //     'required' => false,
                    // ])
                    ->add('firstname', null, ['required' => false])
                    ->add('lastname', null, ['required' => false])
                    // ->add('website', UrlType::class, ['required' => false])
                    // ->add('biography', TextType::class, ['required' => false])
                    // ->add('gender', ChoiceType::class, $genderOptions)
                    // ->add('locale', LocaleType::class, ['required' => false])
                    // ->add('timezone', TimezoneType::class, ['required' => false])
                    ->add('phone', null, ['required' => false])
                ->end()
                // ->with('Social')
                //     ->add('facebookUid', null, ['required' => false])
                //     ->add('facebookName', null, ['required' => false])
                //     ->add('twitterUid', null, ['required' => false])
                //     ->add('twitterName', null, ['required' => false])
                //     ->add('gplusUid', null, ['required' => false])
                //     ->add('gplusName', null, ['required' => false])
                // ->end()
            ->end();
        if (!($subject->isGranted('ROLE_SUPER_ADMIN') || $subject->isGranted('ROLE_ADMIN_SIEGE'))) {
            $formMapper
                ->tab('Security')
                    ->with('Status')
                        ->add('enabled', null, ['required' => false])
                    ->end()
            ;
            $hideOrShowGroupe = ['class' => 'hide'];
            $hideOrShowComptoir = ['class' => 'hide'];
            if (($subject->isGranted('ROLE_GESTION_GROUPE') || $subject->isGranted('ROLE_CONTACT')) && !empty($subject->getGroupesgere())) {
                $hideOrShowGroupe = [];
            }
            if ($subject->isGranted('ROLE_COMPTOIR') && !empty($subject->getComptoirsgere())) {
                $hideOrShowComptoir = [];
            }
            $formMapper
                ->with('Groups')
                    ->add('groups', ModelType::class, [
                        'required' => false,
                        'expanded' => true,
                        'multiple' => true,
                    ])
                    ->add('groupesgere', null, [
                        'required' => false,
                        'label' => 'Groupe local géré (obligatoire)',
                        'attr' => $hideOrShowGroupe,
                        'label_attr' => $hideOrShowGroupe,
                    ])
                    ->add('comptoirsgere', null, [
                        'required' => false,
                        'label' => 'Comptoir géré (obligatoire)',
                        'attr' => $hideOrShowComptoir,
                        'label_attr' => $hideOrShowComptoir,
                    ])
                ->end()
                ->with('Roles')
                    ->add('realRoles', SecurityRolesType::class, [
                        'label' => 'form.label_roles',
                        'expanded' => true,
                        'multiple' => true,
                        'required' => false,
                    ])
                ->end()
            ->end()
            ;
        }
        // ->with('Keys')
        //     ->add('token', null, ['required' => false])
        //     ->add('twoStepVerificationCode', null, ['required' => false])
        // ->end()
    }

    public function getBatchActions()
    {
        $actions = parent::getBatchActions();
        unset($actions['delete']);

        return $actions;
    }

    public function getDashboardActions()
    {
        $actions = parent::getDashboardActions();
        unset($actions['list']);
        return $actions;
    }

    public function preUpdate($user): void
    {
        $em = $this->getConfigurationPool()->getContainer()->get('doctrine')->getManager();

        /* ON EMPECHE ICI DE DESACTIVER LES ROLES SUPER ADMIN OU ADMIN SIEGE */
        if ($user->hasRole('ROLE_SUPER_ADMIN') || $user->hasRole('ROLE_ADMIN_SIEGE')) {
            if (!$user->isEnabled()) {
                $user->setEnabled(1);
                $em->persist($user);
                $em->flush();
            }
        }
    }
}
