<?php

namespace App\Entity;

use App\Entity\EntityTrait\EnablableEntityTrait;
use App\Entity\EntityTrait\HasCompteEntity;
use App\Entity\EntityTrait\NameSlugContentEntityTrait;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity
 * @ORM\Table(name="groupe")
 */
class Groupe
{
    use NameSlugContentEntityTrait,
        TimestampableEntity,
        EnablableEntityTrait,
        HasCompteEntity;

    /**
     * @var int
     *
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * @var Siege
     *
     * @ORM\ManyToOne(targetEntity="Siege", inversedBy="groupes")
     * @ORM\JoinColumn(name="siege_id", referencedColumnName="id", nullable=false)
     */
    private $siege;

    /**
     * @var ArrayCollection|Comptoir[]
     * @ORM\OneToMany(targetEntity="Comptoir", mappedBy="groupe", fetch="EXTRA_LAZY")
     * @ORM\OrderBy({"name": "ASC"})
     */
    private $comptoirs;

    /**
     * @var ArrayCollection|Prestataire[]
     * @ORM\OneToMany(targetEntity="Prestataire", mappedBy="groupe", fetch="EXTRA_LAZY")
     * @ORM\OrderBy({"raison": "ASC"})
     */
    private $prestataires;

    /**
     * @var ArrayCollection|Groupeprestataire[]
     * @ORM\OneToMany(targetEntity="Groupeprestataire", mappedBy="groupe", fetch="EXTRA_LAZY")
     * @ORM\OrderBy({"name": "ASC"})
     */
    private $groupeprestataires;

    /**
     * @var ArrayCollection|Adherent[]
     * @ORM\OneToMany(targetEntity="Adherent", mappedBy="groupe", fetch="EXTRA_LAZY")
     * @ORM\OrderBy({"updatedAt": "ASC"})
     */
    private $adherents;

    /**
     * @var ArrayCollection|User[]
     * @ORM\OneToMany(targetEntity="User", mappedBy="groupesgere", fetch="EXTRA_LAZY")
     */
    private $gestionnaires;

    public function __construct()
    {
        $this->comptoirs = new ArrayCollection();
        $this->prestataires = new ArrayCollection();
        $this->adherents = new ArrayCollection();
        $this->groupeprestataires = new ArrayCollection();
    }

    /**
    * Get siege
    * @return Siege
    */
    public function getSiege()
    {
        return $this->siege;
    }

    /**
    * Set siege
    * @return $this
    */
    public function setSiege($siege)
    {
        $this->siege = $siege;
        return $this;
    }

    /**
     * @return int
     */
    public function getId(): int
    {
        return $this->id;
    }

    /**
     * Set id
     * @param string $id [description]
     */
    public function setId(string $id): self
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @return Comptoir[]|ArrayCollection
     */
    public function getComptoirs()
    {
        return $this->comptoirs;
    }

    /**
     * @param Comptoir $comptoir
     * @return $this
     */
    public function addComptoir(Comptoir $comptoir)
    {
        if (!$this->comptoirs->contains($comptoir)) {
            $this->comptoirs[] = $comptoir;
            $comptoir->setGroupe($this);
        }
        return $this;
    }

    /**
     * @param Comptoir $comptoir
     * @return $this
     */
    public function removeComptoir(Comptoir $comptoir)
    {
        if ($this->comptoirs->contains($comptoir)) {
            $this->comptoirs->removeElement($comptoir);
            $comptoir->setGroupe(null);
        }
        return $this;
    }

    /**
     * @return Prestataire[]|ArrayCollection
     */
    public function getPrestataires()
    {
        return $this->prestataires;
    }

    /**
     * @param Prestataire $prestataire
     * @return $this
     */
    public function addPrestataire(Prestataire $prestataire)
    {
        if (!$this->prestataires->contains($prestataire)) {
            $this->prestataires[] = $prestataire;
            $prestataire->setGroupe($this);
        }
        return $this;
    }

    /**
     * @param Prestataire $prestataire
     * @return $this
     */
    public function removePrestataire(Prestataire $prestataire)
    {
        if ($this->prestataires->contains($prestataire)) {
            $this->prestataires->removeElement($prestataire);
            $prestataire->setGroupe(null);
        }
        return $this;
    }

    /**
     * @return Adherent[]|ArrayCollection
     */
    public function getAdherents()
    {
        return $this->adherents;
    }

    /**
     * @param Adherent $adherent
     * @return $this
     */
    public function addAdherent(Adherent $adherent)
    {
        if (!$this->adherents->contains($adherent)) {
            $this->adherents[] = $adherent;
            $adherent->setGroupe($this);
        }
        return $this;
    }

    /**
     * @param Adherent $adherent
     * @return $this
     */
    public function removeAdherent(Adherent $adherent)
    {
        if ($this->adherents->contains($adherent)) {
            $this->adherents->removeElement($adherent);
            $adherent->setGroupe(null);
        }
        return $this;
    }

    /**
     * @return User[]|ArrayCollection
     */
    public function getGestionnaires()
    {
        return $this->gestionnaires;
    }

    /**
     * @param User $gestionnaire
     * @return $this
     */
    public function addGestionnaire(User $gestionnaire)
    {
        if (!$this->gestionnaires->contains($gestionnaire)) {
            $this->gestionnaires[] = $gestionnaire;
            $gestionnaire->setGestionnaireGroupe($this);
        }
        return $this;
    }

    /**
     * @param User $gestionnaire
     * @return $this
     */
    public function removeGestionnaire(User $gestionnaire)
    {
        if ($this->gestionnaires->contains($gestionnaire)) {
            $this->gestionnaires->removeElement($gestionnaire);
            $gestionnaire->setGestionnaireGroupe(null);
        }
        return $this;
    }

    /**
     * @return Amap[]|ArrayCollection
     */
    public function getGroupeprestataires()
    {
        return $this->groupeprestataires;
    }

    /**
     * @param Amap $amap
     * @return $this
     */
    public function addGroupeprestataire(Groupeprestataire $groupeprestataire)
    {
        if (!$this->groupeprestataires->contains($groupeprestataire)) {
            $this->groupeprestataires[] = $groupeprestataire;
            $groupeprestataire->addGroupe($this);
        }
        return $this;
    }

    /**
     * @param Amap $amap
     * @return $this
     */
    public function removeGroupeprestataire(Groupeprestataire $groupeprestataires)
    {
        if ($this->groupeprestataires->contains($groupeprestataire)) {
            $this->groupeprestataires->removeElement($groupeprestataire);
            $groupeprestataire->removeGroupe($this);
        }
        return $this;
    }


    public function getComptoirsCount()
    {
        return $this->getComptoirs()->count();
    }

    public function getPrestatairesCount()
    {
        return $this->getPrestataires()->count();
    }

    public function getAdherentsCount()
    {
        return $this->getAdherents()->count();
    }

    public function __toString(): string
    {
        return !empty($this->getName())?$this->getName():'Groupe';
    }
}
