<?php

declare(strict_types=1);

namespace App\Command;

use App\Entity\Flux;
use App\Entity\GlobalParameter;
use App\Entity\Prestataire;
use App\Entity\Reconversion;
use App\Entity\Siege;
use App\Enum\MoyenEnum;
use App\Utils\CustomEntityManager;
use App\Utils\OperationUtils;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Twig\Environment;

/**
 * This command is part of the CCAS reconversion process.
 * @see SendCcasTransactionsExportToPrestataire for more comments.
 *
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 */
class ReconversionCcasMonaPrestatairesCommand extends Command
{
    protected static $defaultName = 'kohinos:ssa:reconversion-ccas-prestataires';

    protected $em;
    protected $mailer;
    protected $templating;
    protected $io;
    protected $param;
    protected $operationUtils;

    public function __construct(
        CustomEntityManager $em,
        \Swift_Mailer $mailer,
        Environment $templating,
        OperationUtils $operationUtils
    ) {
        $this->em = $em;
        $this->mailer = $mailer;
        $this->templating = $templating;
        $this->operationUtils = $operationUtils;
        parent::__construct();
    }

    protected function configure()
    {
        $this->setDescription('SSA : générer les flux de reconversion ccas du mois précédent');
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->io = new SymfonyStyle($input, $output);

        $this->io->title('START. Reconversions CCAS');

        $prestas = $this->em->getRepository(Prestataire::class)->findAll();

        $firstDayOfPreviousMonth = new \DateTime('first day of previous month');

        foreach ($prestas as $p) {
            $sumOfCcasTransactionsOfPreviousMonth = $this->em->getRepository(Flux::class)->getSumOfValidCcasTransactionsByPrestaAndMonth(
                $p,
                intval($firstDayOfPreviousMonth->format('m')),
                $firstDayOfPreviousMonth->format('Y'),
            );

            $montant = round(floatval($sumOfCcasTransactionsOfPreviousMonth),2);

            if ($montant <= 0) {
                continue;
            }
            /* @var Prestataire $p */
            $flux = new Reconversion();
            $flux->setReconversionCcas(true);
            //do not fill operator as it is automated process $entity->setOperateur();
            $flux->setExpediteur($p);
            $flux->setMontant($montant);
            $flux->setDestinataire($this->em->getRepository(Siege::class)->getTheOne());
            $flux->setMoyen(MoyenEnum::MOYEN_AUTRE);
            $flux->setTauxreconversion(
                !empty($p->getTauxreconversion()) ?
                    $p->getTauxreconversion()
                    : floatval(
                        $this->em->getRepository(GlobalParameter::class)->val(GlobalParameter::RECONVERSION_PRESTATAIRE)
                    )
            );
            $now = new \DateTime();
            $flux->setReference(
                'Reconversion CCAS automatique du ' . $now->format('d/m/Y')
            );
            $this->operationUtils->executeOperations($flux);

        }

        $this->io->success('End');

        $memoryUsage = memory_get_usage(true) / 1024 / 1024;
        $this->io->text("Batch finished with memory: ${memoryUsage}M");

        return 0;
    }
}
