<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Doctrine\UuidGenerator;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;

/**
 * @ORM\Entity(repositoryClass="App\Repository\ExternalAdherentDatedDataRepository")
 * @ORM\Table(name="external_adherent_dated_data", uniqueConstraints={@ORM\UniqueConstraint(name="adherentyear", columns={"adherent_id", "year"})})
 * @UniqueEntity(
 *     fields={"adherent", "year"},
 *     errorPath="adherent",
 *     message="Une entrée existe déjà pour cet adhérent et cette année"
 * )
 */
class ExternalAdherentDatedData
{
    /**
     * @var \Ramsey\Uuid\UuidInterface
     *
     * @ORM\Id
     * @ORM\Column(type="uuid", unique=true)
     * @ORM\GeneratedValue(strategy="CUSTOM")
     * @ORM\CustomIdGenerator(class=UuidGenerator::class)
     */
    private $id;

    /**
     * @ORM\ManyToOne(targetEntity=Adherent::class, inversedBy="externalDatedData")
     * @ORM\JoinColumn(name="adherent_id", referencedColumnName="id", nullable=false)
     */
    private $adherent;

    /** @ORM\Column(type="integer") */
    private $year;

    /** @ORM\Column(type="float", nullable=true) */
    private $householdUnitShares;

    /** @ORM\Column(type="integer", nullable=true) */
    private $annualIncome;

    /** @ORM\Column(type="integer", nullable=true) */
    private $monthlyIncome;

    public function setAdherent(Adherent $adherent): self
    {
        $this->adherent = $adherent;

        return $this;
    }

    public function setYear(int $year): self
    {
        $this->year = $year;

        return $this;
    }

    public function setHouseholdUnitShares(?float $householdUnitShares): self
    {
        $this->householdUnitShares = $householdUnitShares;

        return $this;
    }

    public function setAnnualIncome(?int $annualIncome): self
    {
        $this->annualIncome = $annualIncome;

        return $this;
    }

    public function setMonthlyIncome(?int $monthlyIncome): self
    {
        $this->monthlyIncome = $monthlyIncome;

        return $this;
    }

    public function getAdherent(): Adherent
    {
        return $this->adherent;
    }

    public function getYear(): ?int
    {
        return $this->year;
    }

    public function getHouseholdUnitShares(): ?float
    {
        return $this->householdUnitShares;
    }

    public function getAnnualIncome(): ?int
    {
        return $this->annualIncome;
    }

    public function getMonthlyIncome(): ?int
    {
        return $this->monthlyIncome;
    }

    public function __toString(): string
    {
        $incomes = [];
        if ($this->getMonthlyIncome() != null) {
            $incomes[] = "mensuels : " . $this->getMonthlyIncome() . "€";
        }
        if ($this->getAnnualIncome() != null) {
            $incomes[] = "annuels : " . $this->getAnnualIncome() . "€";
        }

        if (count($incomes) > 0) {
            return "Année " . $this->getYear() . " : Revenus " . join(" / ", $incomes);
        } else {
            return "";
        }
    }

}
