<?php

namespace App\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use App\Entity\EntityTrait\EnablableEntityTrait;
use App\Entity\EntityTrait\GeolocEntityTrait;
use App\Entity\EntityTrait\HasAccountsTrait;
use App\Entity\EntityTrait\HasEcompteEntity;
use App\Flux\AccountableInterface;
use App\Flux\AccountableObject;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Ramsey\Uuid\Doctrine\UuidGenerator;
use Symfony\Component\Serializer\Annotation\Groups;

/**
 * ApiResource(
 *     attributes={"security"="is_granted('ROLE_ADMIN_ADHERENT_GERER_VIEW')"},
 *     collectionOperations={
 *         "get"={"security"="is_granted('ROLE_ADMIN_ADHERENT_GERER_LIST')"},
 *         "post"={"security"="is_granted('ROLE_ADMIN_ADHERENT_GERER_EDIT')"}
 *     },
 *     itemOperations={
 *         "get"={"security"="is_granted('ROLE_ADMIN_ADHERENT_GERER_LIST')"},
 *         "put"={"security"="is_granted('ROLE_ADMIN_ADHERENT_GERER_EDIT') or object.user == user"},
 *     },
 *     normalizationContext={"groups"={"read"}},
 *     denormalizationContext={"groups"={"write"}}
 * ).
 *
 * @ORM\Entity(repositoryClass="App\Repository\AdherentRepository")
 * @ORM\Table(name="adherent")
 * @ORM\HasLifecycleCallbacks()
 */
class Adherent extends AccountableObject implements AccountableInterface
{
    use EnablableEntityTrait;
    use TimestampableEntity;
    use GeolocEntityTrait;
    use HasEcompteEntity;
    use HasAccountsTrait;

    /**
     * @var \Ramsey\Uuid\UuidInterface
     *
     * @ORM\Id
     * @ORM\Column(type="uuid", unique=true)
     * @ORM\GeneratedValue(strategy="CUSTOM")
     * @ORM\CustomIdGenerator(class=UuidGenerator::class)
     */
    protected $id;

    /**
     * @var string
     *
     * @ORM\Column(name="idmlc", type="string", length=100, nullable=true)
     * @Groups({"read", "write"})
     */
    protected $idmlc;

    /**
     * @var User
     *
     * @ORM\OneToOne(targetEntity="User", cascade={"all"}, mappedBy="adherent", fetch="LAZY")
     */
    protected $user;

    /**
     * @var Groupe
     *
     * @ORM\ManyToOne(targetEntity="Groupe", inversedBy="adherents")
     */
    private $groupe;

    /**
     * @var ArrayCollection|AccountAdherent[]
     * @ORM\OneToMany(targetEntity="AccountAdherent", mappedBy="adherent", cascade={"remove"})
     */
    private $accounts;

    /**
     * @ORM\Column(type="string", length=255, nullable=true)
     */
    private $paymentCode;

    /**
     * @var ProfilDeCotisation
     *
     * @ORM\ManyToOne(targetEntity="ProfilDeCotisation", cascade={"persist"}, inversedBy="beneficiaires")
     * @ORM\JoinColumn(name="profildecotisation_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
     */
    private $profilDeCotisation;

    /**
     * @ORM\Column(type="string", length=255, nullable=true)
     */
    private $moyenDePaiement;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private $jourPrelevement;

    /**
     * @ORM\Column(type="boolean", nullable=true)
     */
    private $mailRappelCotisation;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private $jourMailRappelCotisation;

    /**
     * @ORM\OneToMany(targetEntity=DependentChild::class, mappedBy="adherent", orphanRemoval=true, cascade={"persist"})
     */
    private $dependentChildren;

    /**
     * @ORM\Column(type="string", length=255, nullable=true)
     */
    private $householdComposition;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private $householdAdultCount;

    /**
     * In simplified household based allocation process, don't keep track of adults & children in the household 
     * 
     * @ORM\Column(type="integer", nullable=true)
     */
    private $householdCount;

    /**
     * On household based allowance process, define a cotisation amount for each adherent.
     * 
     * @ORM\Column(type="float", nullable=true)
     */
    private $cotisationAmount;

    /**
     * On household based allowance process, the allowance amount is calculated based on household data.
     * Calculate and save the allocation amount when the household data is updated.
     * 
     * @ORM\Column(type="float", nullable=true)
     */
    private $allocationAmount;

    /**
     * @ORM\Column(type="boolean", options={"default": false})
     */
    private $ccasAccepted = false;

    /**
     * @ORM\Column(type="boolean", options={"default": false})
     */
    private $ccasEligible = false;

    /**
     * @ORM\Column(type="string", length=36, nullable=true, unique=true)
     */
    private $anonymousToken;

    /**
     * @ORM\OneToOne(targetEntity="ExternalAdherentData", cascade={"persist", "remove"}, mappedBy="adherent", orphanRemoval=true)
     */
    private ?ExternalAdherentData $externalData = null;

    /**
     * @ORM\OneToMany(targetEntity=ExternalAdherentDatedData::class, mappedBy="adherent", orphanRemoval=true, cascade={"persist"})
     */
    private $externalDatedDataCollection;

    public function __construct()
    {
        $this->accounts = new ArrayCollection();
        $this->dependentChildren = new ArrayCollection();
        $this->externalDatedDataCollection = new ArrayCollection();
    }

    public function getId()
    {
        return $this->id;
    }

    /**
     * Get idmlc.
     *
     * @return
     */
    public function getIdmlc(): ?string
    {
        return $this->idmlc;
    }

    /**
     * Set idmlc.
     *
     * @return $this
     */
    public function setIdmlc(string $idmlc): self
    {
        $this->idmlc = $idmlc;

        return $this;
    }

    public function getCompte(): float
    {
        return $this->getEcompte();
    }

    public function setCompte(float $ecompte): self
    {
        $this->setEcompte($ecompte);

        return $this;
    }

    /**
     * @return User
     */
    public function getUser(): ?User
    {
        return $this->user;
    }

    /**
     * @param User $user
     *
     * @return Prestataire
     */
    public function setUser(User $user): self
    {
        $this->user = $user;

        return $this;
    }

    /**
     * @param Groupe|null $groupe
     *
     * @return $this
     */
    public function setGroupe(?Groupe $groupe): self
    {
        $this->groupe = $groupe;

        return $this;
    }

    /**
     * @return Groupe|null
     */
    public function getGroupe(): ?Groupe
    {
        return $this->groupe;
    }

    public function isEnabled(): bool
    {
        return $this->getUser()->isEnabled();
    }

    public function setEnabled($enabled): self
    {
        $this->getUser()->setEnabled($enabled);

        return $this;
    }

    public function getName(): string
    {
        return $this->__toString();
    }

    public function __toString(): string
    {
        if (!empty($this->getUser())) {
            if (!empty($this->getUser()->getLastname() . $this->getUser()->getFirstname())) {
                return $this->getUser()->getLastname() . ' ' . $this->getUser()->getFirstname();
            }
            if (!empty($this->getUser()->getUsername())) {
                return $this->getUser()->getUsername();
            }
            if (!empty($this->getUser()->getEmail())) {
                return $this->getUser()->getEmail();
            }
        }

        return 'Adhérent xxx';
    }

    public function getEmail(): ?string
    {
        if (!empty($this->getUser())) {
            return $this->getUser()->getEmail();
        }
        return '';
    }

    public function getFullname(): string
    {
        $return = '';
        if (!empty($this->getUser())) {
            if (!empty($this->getUser()->getLastname() . $this->getUser()->getFirstname())) {
                $return = $this->getUser()->getLastname() . ' ' . $this->getUser()->getFirstname();
                if (!empty($this->getUser()->getEmail())) {
                    $return .= '(' . $this->getUser()->getEmail() . ')';
                }
            } elseif (!empty($this->getUser()->getEmail())) {
                $return = $this->getUser()->getEmail();
            }

            return $return;
        }

        return 'Adhérent xxx';
    }

    public function getPaymentCode(): ?string
    {
        return $this->paymentCode;
    }

    public function setPaymentCode(?string $paymentCode): self
    {
        $this->paymentCode = $paymentCode;

        return $this;
    }

    /**
     * @return ProfilDeCotisation
     */
    public function getProfilDeCotisation(): ?ProfilDeCotisation
    {
        return $this->profilDeCotisation;
    }

    /**
     * @return $this
     */
    public function setProfilDeCotisation($profilDeCotisation): self
    {
        $this->profilDeCotisation = $profilDeCotisation;

        return $this;
    }

    public function getMoyenDePaiement(): ?string
    {
        return $this->moyenDePaiement;
    }

    public function setMoyenDePaiement(?string $moyenDePaiement): self
    {
        $this->moyenDePaiement = $moyenDePaiement;

        return $this;
    }
    
    public function getHouseholdComposition(): ?string
    {
        return $this->householdComposition;
    }

    public function setHouseholdComposition(?string $householdComposition): self
    {
        $this->householdComposition = $householdComposition;

        return $this;
    }

    public function getJourPrelevement(): ?int
    {
        return $this->jourPrelevement;
    }

    public function setJourPrelevement(?int $jourPrelevement): self
    {
        $this->jourPrelevement = $jourPrelevement;

        return $this;
    }
    
    public function getHouseholdAdultCount(): ?int
    {
        return $this->householdAdultCount;
    }

    public function setHouseholdAdultCount(?int $householdAdultCount): self
    {
        $this->householdAdultCount = $householdAdultCount;

        return $this;
    }

    public function getHouseholdCount(): ?int
    {
        return $this->householdCount;
    }

    public function setHouseholdCount(?int $householdCount): self
    {
        $this->householdCount = $householdCount;

        return $this;
    }

    public function getMailRappelCotisation(): ?bool
    {
        return $this->mailRappelCotisation;
    }

    public function setMailRappelCotisation(?bool $mailRappelCotisation): self
    {
        $this->mailRappelCotisation = $mailRappelCotisation;

        return $this;
    }

    public function getJourMailRappelCotisation(): ?int
    {
        return $this->jourMailRappelCotisation;
    }

    public function setJourMailRappelCotisation(?int $jourMailRappelCotisation): self
    {
        $this->jourMailRappelCotisation = $jourMailRappelCotisation;

        return $this;
    }

    /**
     * @return Collection<int, DependentChild>
     */
    public function getDependentChildren(): Collection
    {
        return $this->dependentChildren;
    }

    public function addDependentChild(DependentChild $dependentChild): self
    {
        if (!$this->dependentChildren->contains($dependentChild)) {
            $this->dependentChildren[] = $dependentChild;
            $dependentChild->setAdherent($this);
        }

        return $this;
    }

    public function removeDependentChild(DependentChild $dependentChild): self
    {
        if ($this->dependentChildren->removeElement($dependentChild)) {
            // set the owning side to null (unless already changed)
            if ($dependentChild->getAdherent() === $this) {
                $dependentChild->setAdherent(null);
            }
        }

        return $this;
    }

    public function getCotisationAmount(): ?float
    {
        return $this->cotisationAmount;
    }

    public function setCotisationAmount(?float $cotisationAmount): self
    {
        $this->cotisationAmount = $cotisationAmount;

        return $this;
    }

    public function getAllocationAmount(): ?float
    {
        return $this->allocationAmount;
    }

    public function setAllocationAmount(?float $allocationAmount): self
    {
        $this->allocationAmount = $allocationAmount;

        return $this;
    }

    public function getCeiling()
    {
        //This formula has been configured for ssa gironde project
        return 2 * $this->allocationAmount;
    }

    public function getCcasAccepted(): ?bool
    {
        return $this->ccasAccepted;
    }

    public function setCcasAccepted(bool $ccasAccepted): self
    {
        $this->ccasAccepted = $ccasAccepted;

        return $this;
    }

    public function getCcasEligible(): ?bool
    {
        return $this->ccasEligible;
    }

    public function setCcasEligible(bool $ccasEligible): self
    {
        $this->ccasEligible = $ccasEligible;

        return $this;
    }

    public function getAnonymousToken(): ?string
    {
        return $this->anonymousToken;
    }

    public function setAnonymousToken(?string $anonymousToken): self
    {
        $this->anonymousToken = $anonymousToken;

        return $this;
    }

    /**
     * @return ExternalAdherentData
     */
    public function getExternalData(): ?ExternalAdherentData
    {
        return $this->externalData;
    }

    /**
     * @param ExternalAdherentData $externalData
     *
     * @return Adherent
     */
    public function setExternalData(?ExternalAdherentData $externalData): self
    {
        $this->externalData = $externalData;

        return $this;
    }

    /**
     * @return Collection<int, ExternalAdherentDatedData>
     */
    public function getExternalDatedDataCollection(): Collection
    {
        return $this->externalDatedDataCollection;
    }

    public function addExternalDatedDataCollection(ExternalAdherentDatedData $externalDatedData): self
    {
        if (!$this->externalDatedDataCollection->contains($externalDatedData)) {
            $this->externalDatedDataCollection[] = $externalDatedData;
            $externalDatedData->setAdherent($this);
        }

        return $this;
    }

    public function removeExternalDatedDataCollection(ExternalAdherentDatedData $externalDatedData): self
    {
        $this->externalDatedDataCollection->removeElement($externalDatedData);
        return $this;
    }
}
