<?php

declare(strict_types=1);

namespace App\Command;

use App\Entity\Adherent;
use App\Entity\Flux;
use App\Entity\GlobalParameter;
use App\Entity\Prestataire;
use App\Entity\SolidoumeItem;
use App\Entity\SolidoumeParameter;
use App\Entity\TransactionAdherentPrestataire;
use App\Entity\TransactionPrestataireAdherent;
use App\Entity\User;
use App\Enum\CurrencyEnum;
use App\Enum\MoyenEnum;
use App\Utils\CustomEntityManager;
use App\Utils\OperationUtils;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Twig\Environment;

/**
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 */
class SendMailRappelCotisationCommand extends Command
{
    protected static $defaultName = 'kohinos:tav:mail-rappel-cotisation';

    protected $em;
    protected $mailer;
    protected $templating;
    protected $io;
    protected $param;
    protected $operationUtils;

    public function __construct(
        CustomEntityManager $em,
        \Swift_Mailer $mailer,
        Environment $templating,
        OperationUtils $operationUtils
    ) {
        $this->em = $em;
        $this->mailer = $mailer;
        $this->templating = $templating;
        $this->operationUtils = $operationUtils;
        parent::__construct();
    }

    protected function configure()
    {
        $this->setDescription('TAV : envoyer les mails de rappel des cotisations');
    }

    /**
     *
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->io = new SymfonyStyle($input, $output);
        $this->io->title('Start');

        $this->executeReminders();

        $this->io->success('End');

        $memoryUsage = memory_get_usage(true) / 1024 / 1024;
        $this->io->text("Batch finished with memory: ${memoryUsage}M");

        return 0;
    }

    private function executeReminders()
    {
        $this->io->title('START : Rappel par email');

        $today = new \DateTime();
        $jour = $today->format('d');

        $adherents = $this->em->getRepository(Adherent::class)->findBy(array(
            'mailRappelCotisation' => true,
            'jourMailRappelCotisation' => $jour
        ));

        foreach ($adherents as $adherent) {
            $this->sendReminder($adherent);
        }
    }

    /**
     * Send email reminder.
     *
     * @param Adherent $adherent
     */
    private function sendReminder(Adherent $adherent)
    {
        $this->io->success("Envoi de l'email de rappel à l'adhérent : " . $adherent->__toString());

        $subject = '[MONNAIE ALIMENTAIRE COMMUNE] – Rappel automatique cotisation';
        $mail = (new \Swift_Message($subject))
            ->setFrom($this->em->getRepository(GlobalParameter::class)->val(GlobalParameter::MLC_NOTIF_EMAIL))
            ->setTo($adherent->getUser()->getEmail())
            ->setBody(
                $this->templating->render(
                    '@kohinos/email/tav/rappel_cotisation.html.twig',
                    [
                        'subject' => $subject
                    ]
                ),
                'text/html'
            );
        $this->mailer->send($mail);
    }
}