<?php

namespace App\Entity;

use ApiPlatform\Core\Annotation\ApiResource;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Ramsey\Uuid\Doctrine\UuidGenerator;
use Symfony\Component\Validator\Constraints as Assert;


/**
 * @ORM\Entity(repositoryClass="App\Repository\ProfilDeCotisationRepository")
 * @ORM\HasLifecycleCallbacks()
 * @ORM\Table(name="profildecotisation")
 */
class ProfilDeCotisation
{
    use TimestampableEntity;

    /**
     * @var \Ramsey\Uuid\UuidInterface
     *
     * @ORM\Id
     * @ORM\Column(type="uuid", unique=true)
     * @ORM\GeneratedValue(strategy="CUSTOM")
     * @ORM\CustomIdGenerator(class=UuidGenerator::class)
     */
    protected $id;

    /**
     * The amount the user will pay.
     * The amount the user will receive depends on tauxCotisation.
     * 
     * @ORM\Column(name="montant", type="integer")
     * @Assert\Type("numeric")
     * @Assert\GreaterThanOrEqual(
     *     value = 0
     * )
     */
    protected $montant;

    /**
     * @var float
     *
     * @ORM\Column(name="tauxCotisation", type="decimal", scale=4)
     * @Assert\NotBlank
     * @Assert\Type("numeric")
     * @Assert\GreaterThanOrEqual(
     *     value = 0
     * )
     */
    protected $tauxCotisation;

    /**
     * @var ArrayCollection|Adherent[]
     * @ORM\OneToMany(targetEntity="Adherent", cascade={"persist"}, mappedBy="profildecotisation")
     */
    private $beneficiaires;

    public function __construct()
    {
        $this->beneficiaires = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Get tauxCotisation.
     *
     * @return float tauxCotisation
     */
    public function getTauxCotisation(): ?float
    {
        return $this->tauxCotisation;
    }

    /**
     * Set tauxCotisation.
     *
     * @return $this
     */
    public function setTauxCotisation(?float $tauxCotisation): self
    {
        $this->tauxCotisation = $tauxCotisation;

        return $this;
    }

    /**
     * Get montant.
     *
     * @return int montant
     */
    public function getMontant(): ?int
    {
        return $this->montant;
    }

    /**
     * Set montant.
     *
     * @return $this
     */
    public function setMontant(?int $montant): self
    {
        $this->montant = $montant;

        return $this;
    }

    /**
     * @return ArrayCollection
     */
    public function getBeneficiaires()
    {
        return $this->beneficiaires;
    }

    /**
     * setBeneficiaires.
     *
     * @param [type] $beneficiaires [description]
     */
    public function setBeneficiaires($beneficiaires): self
    {
        $this->beneficiaires = $beneficiaires;

        return $this;
    }

    /**
     * @param Adherent $beneficiaire
     *
     * @return $this
     */
    public function addBeneficiaire(Adherent $beneficiaire): self
    {
        if (is_null($this->beneficiaires)) {
            $this->beneficiaires = new ArrayCollection();
        }
        if (!$this->beneficiaires->contains($beneficiaire)) {
            $this->beneficiaires[] = $beneficiaire;
            $beneficiaire->setProfilDeCotisation($this);
        }

        return $this;
    }

    /**
     * @param Adherent $beneficiaire
     *
     * @return $this
     */
    public function removeBeneficiaire(Adherent $beneficiaire): self
    {
        if ($this->beneficiaires->contains($beneficiaire)) {
            $this->beneficiaires->removeElement($beneficiaire);
            $beneficiaire->setProfilDeCotisation(null);
        }

        return $this;
    }

    public function __toString(): string
    {
        return 'Montant ' . strval($this->montant) . '€'
            . ', Taux ' . strval(number_format($this->tauxCotisation, 2, '.', ''));
    }
}
