<?php

namespace App\Controller;

use App\Entity\Caissier;
use App\Entity\Groupe;
use App\Entity\Prestataire;
use App\Entity\Rubrique;
use App\Entity\Flux;
use App\Entity\SelfEvalPrestaQuiz;
use App\Form\Type\DistributorSelfEvalPrestaQuizType;
use App\Form\Type\ProducerSelfEvalPrestaQuizType;
use App\Form\Type\SelfEvalPrestaQuizType;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Core\Security;

class PrestatairesController extends FrontController
{
    protected $em;
    private $router;
    private $session;
    private $security;

    public function __construct(EntityManagerInterface $em, RouterInterface $router, SessionInterface $session, Security $security)
    {
        $this->em = $em;
        $this->router = $router;
        $this->session = $session;
        $this->security = $security;
    }

    /**
     * @Route("/prestataire/{slug}", name="show_prestataire")
     */
    public function showPrestaAction(Prestataire $prestataire)
    {
        if (!$this->isFrontActivated()) {
            return $this->redirectToRoute('index');
        }
        if ($prestataire->isMlc()) {
            // Ne pas montrer la page du prestataire recevant les cotisations !
            return new RedirectResponse($this->router->generate('index'));
        }

        $templateData = ['presta' => $prestataire];

        // If feature activated, display self evaluation form 
        if ($this->getParameter('presta_self_init_and_eval') == true) {
            $quiz = $prestataire->getSelfEvalPrestaQuiz();

            if ($quiz->isSubmitted) {
                $formClass = Prestataire::DISTRIBUTOR === $prestataire->getMarketChannelFunction() ?
                    DistributorSelfEvalPrestaQuizType::class : ProducerSelfEvalPrestaQuizType::class;
                $form = $this->createForm($formClass, $quiz, ["mode" => SelfEvalPrestaQuizType::READONLY]);
    
                $templateData['form'] = $form->createView();
            }
        }

        return $this->render('@kohinos/presta/show.html.twig', $templateData);
    }

    /**
     * @Route("/prestataires/liste/{order}", name="liste_prestataire", defaults={"order": "raison"})
     */
    public function listePrestaAction($order = 'raison')
    {
        if (!$this->isFrontActivated()) {
            return $this->redirectToRoute('index');
        }

        if ('groupelocal' == $order) {
            $prestas = $this->em->getRepository(Prestataire::class)->findDefault('prestataire', 'groupelocal');

            return $this->render('@kohinos/presta/liste_prestataires_bygroupelocal.html.twig', [
                'prestas' => $prestas,
                'type' => 'Prestataires',
            ]);
        }
        $prestas = $this->em->getRepository(Prestataire::class)->findDefault('prestataire');

        return $this->render('@kohinos/presta/liste_prestataires.html.twig', [
            'prestas' => $prestas,
            'type' => 'Prestataires',
        ]);
    }

    /**
     * @Route("/prestataires/groupe/{slug}/liste", name="liste_presta_by_groupe")
     */
    public function listePrestaByGroupeAction(Groupe $groupe)
    {
        if (!$this->isFrontActivated()) {
            return $this->redirectToRoute('index');
        }

        $prestas = $this->em->getRepository(Prestataire::class)->findByGroupe($groupe);

        return $this->render('@kohinos/presta/liste_prestataires.html.twig', [
            'prestas' => $prestas,
            'type' => 'Prestataires du groupe local ' . $groupe->getName(),
        ]);
    }

    /**
     * @Route("/partenaires/liste", name="liste_partenaire")
     */
    public function listePartnerAction()
    {
        if (!$this->isFrontActivated()) {
            return $this->redirectToRoute('index');
        }
        $partners = $this->em->getRepository(Prestataire::class)->findDefault('partenaire');

        return $this->render('@kohinos/presta/liste_prestataires.html.twig', [
            'prestas' => $partners,
            'type' => 'Partenaires',
        ]);
    }

    /**
     * @Route("/prestataires/carte", name="carte_prestataire")
     */
    public function cartePrestaAction()
    {
        if (!$this->isFrontActivated()) {
            return $this->redirectToRoute('index');
        }
        $prestas = $this->em->getRepository(Prestataire::class)->findDefault();

        return $this->render('@kohinos/presta/carte.html.twig', [
            'prestas' => $prestas,
        ]);
    }

    /**
     * @Route("/prestataires/rubriques", name="rubriques_prestataire")
     */
    public function rubriquesAction()
    {
        if (!$this->isFrontActivated()) {
            return $this->redirectToRoute('index');
        }
        $rubriques = $this->em->getRepository(Rubrique::class)->findBy(['enabled' => true], ['name' => 'ASC']);

        return $this->render('@kohinos/presta/rubriques.html.twig', [
            'rubriques' => $rubriques,
        ]);
    }

    /**
     * @Route("/prestataires/rubrique/{slug}", name="show_rubrique")
     */
    public function showRubriqueAction(Rubrique $rubrique)
    {
        if (!$this->isFrontActivated()) {
            return $this->redirectToRoute('index');
        }

        return $this->render('@kohinos/presta/show_rubrique.html.twig', [
            'rubrique' => $rubrique,
            'prestataires' => $this->em->getRepository(Prestataire::class)->findByRubrique($rubrique),
        ]);
    }

    /**
     * Get the total transactions amount towards the Prestataire operated by the connected Caissier,
     * since the last time it was fetched.
     * 
     * @Route("/prestataires/get_caissier_last_transactions", name="get_caissier_last_transactions")
     * @IsGranted({"ROLE_CAISSIER"})
     */
    public function getLastCaissierTransactionsAmount()
    {
        if (!$this->session->has('_prestagere')) {
            return null;
        }

        $user = $this->security->getUser();
        $presta = $this->em->getRepository(Prestataire::class)->findOneById($this->session->get('_prestagere')->getId());
        $caissier = $this->em->getRepository(Caissier::class)->findOneBy(['prestataire' => $presta, 'user' => $user]);

        // Get last caissier export datetime
        $datetime_last_export = $caissier->getLastTransactionsExportDatetime();
        if (null == $datetime_last_export) {
            // for transition purposes from Prestataire level export towards Caissier level export,
            // get prestataire last export if caissier export date is not yet set
            $datetime_last_export = $caissier->getLastTransactionsExportDatetime();

            if (null == $datetime_last_export) {
                // prestaire creation if first export
                $datetime_last_export = $presta->getCreatedAt();
            }
        }

        // Get total amount of transactions operated by connecteed Caissier for this Prestataire
        $flux = $this->em->getRepository(Flux::class)->getQueryByCaissier(
            $user,
            $presta,
            $datetime_last_export->format(("Y-m-d H:i:s"))
        )->getResult();

        $total_amount = 0;
        foreach ($flux as $flux_item) {
            // Exclude reconversions from calculation (extra security but probably unecessary, as reconversions don't have an operator)
            if ($flux_item->getType() != "reconversion_prestataire") {
                $total_amount += $flux_item->getMontant();
            }
        }

        // Set now as this caissier last export datetime
        $caissier->setLastTransactionsExportDatetime(new \Datetime('now'));
        $this->em->persist($caissier);
        $this->em->flush();

        $str_datetime = $datetime_last_export->format('d/m/Y H\hi');

        return $this->render('@kohinos/tav/last_transactions_amount.html.twig', [
            'amount' => $total_amount,
            'datetime_last_export' => $str_datetime
        ]);
    }
}
