<?php

namespace App\Entity;

use App\Repository\AllocationRepository;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Entity(repositoryClass=AllocationRepository::class)
 * 
 * In SSA environment, defines the allocation an Adhérent gets when they pay their cotisation.
 * An allocation is composed of a cotisation, which is the amount they will pay,
 * and an optional complementary flux to add to or withdraw from the user eMLC account so they get an allocation of [amount] eMLC.
 * 
 * This table is primarily used in an external context (i.e. Metabase), to simplify access to allocation data.
 */
class Allocation
{
    use TimestampableEntity;

    /**
     * @ORM\Id
     * @ORM\GeneratedValue
     * @ORM\Column(type="integer")
     */
    private $id;

    /**
     * @ORM\ManyToOne(targetEntity=adherent::class, inversedBy="allocations")
     * @ORM\JoinColumn(nullable=false)
     */
    private $adherent;

    /**
     * @ORM\OneToOne(targetEntity=Flux::class, inversedBy="allocation", cascade={"persist", "remove"})
     * @ORM\JoinColumn(nullable=false)
     */
    private $cotisationFlux;

    /**
     * @ORM\OneToOne(targetEntity=Flux::class, inversedBy="allocation", cascade={"persist", "remove"})
     * 
     * Can be null if the user gets the same amount of allocation as their cotisation.
     */
    private $complementaryFlux;

    /**
     * @ORM\Column(type="float")
     * 
     * Amount of the allocation, combinaison of cotisation flux to which is added or added the complementary flux
     */
    private $amount;

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getAdherent(): ?adherent
    {
        return $this->adherent;
    }

    public function setAdherent(?adherent $adherent): self
    {
        $this->adherent = $adherent;

        return $this;
    }

    public function getCotisationFlux(): ?Flux
    {
        return $this->cotisationFlux;
    }

    public function setCotisationFlux(Flux $cotisationFlux): self
    {
        $this->cotisationFlux = $cotisationFlux;

        return $this;
    }

    public function getComplementaryFlux(): ?Flux
    {
        return $this->complementaryFlux;
    }

    public function setComplementaryFlux(?Flux $complementaryFlux): self
    {
        $this->complementaryFlux = $complementaryFlux;

        return $this;
    }

    public function getAmount(): ?float
    {
        return $this->amount;
    }

    public function setAmount(float $amount): self
    {
        $this->amount = $amount;

        return $this;
    }
}
