<?php

declare(strict_types=1);

namespace App\Command;

use App\Entity\GlobalParameter;
use App\Entity\Prestataire;
use App\Entity\Reconversion;
use App\Entity\Siege;
use App\Enum\MoyenEnum;
use App\Enum\ReconversionFrequencyEnum;
use App\Utils\CustomEntityManager;
use App\Utils\OperationUtils;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Twig\Environment;

/**
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 */
class ReconversionMonaPrestatairesCommand extends Command
{
    protected static $defaultName = 'kohinos:ssa:reconversion-prestataires';

    protected $em;
    protected $mailer;
    protected $templating;
    protected $io;
    protected $param;
    protected $operationUtils;

    public function __construct(
        CustomEntityManager $em,
        \Swift_Mailer $mailer,
        Environment $templating,
        OperationUtils $operationUtils
    ) {
        $this->em = $em;
        $this->mailer = $mailer;
        $this->templating = $templating;
        $this->operationUtils = $operationUtils;
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setDescription('SSA : générer des flux de reconversion conformément à la fréquence configurée par les prestataires')
            ->addArgument('reconvmode', InputArgument::REQUIRED, 'Restriction aux seuls prestataires ayant opté pour la fréquence choisie.');
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->io = new SymfonyStyle($input, $output);
        $reconvmode = $input->getArgument('reconvmode');

        if (!in_array($reconvmode, ReconversionFrequencyEnum::getAvailableTypes())) {
            $this->io->error("L'argument " . $reconvmode . ' est invalide. Choix possibles : '
                . implode(', ', ReconversionFrequencyEnum::getAvailableTypes()));
            return 1;
        }

        $this->io->title('START. Reconversion pour les prestataires de compte positif ayant choisi la fréquence : ' . $reconvmode);

        $prestas = $this->em->getRepository(Prestataire::class)->findBy(['reconversionFrequency' => $reconvmode]);
        foreach ($prestas as $p) {
            //in CCAS_MODE, ccas transactions reconversions are processed separately.
            //getBalanceCcas will return 0 if CCAS_MODE is disabled
            $balance = $p->getEmlcAccount()->getBalance() - $p->getEmlcAccount()->getBalanceCcas();
            if ($balance <= 0) {
                continue;
            }
            /* @var Prestataire $p */
            $flux = new Reconversion();
            //do not fill operator as it is automated process $entity->setOperateur();
            $flux->setExpediteur($p);
            $flux->setMontant($balance); //montant maximal
            $flux->setDestinataire($this->em->getRepository(Siege::class)->getTheOne());
            $flux->setMoyen(MoyenEnum::MOYEN_AUTRE);
            $flux->setTauxreconversion(
                !empty($p->getTauxreconversion()) ?
                    $p->getTauxreconversion()
                    : floatval(
                        $this->em->getRepository(GlobalParameter::class)->val(GlobalParameter::RECONVERSION_PRESTATAIRE)
                    )
            );
            $now = new \DateTime();
            $flux->setReference(
                'Reconversion automatique du ' . $now->format('d/m/Y')
                . " pour le mode de reconversion " . $reconvmode
            );
            $this->operationUtils->executeOperations($flux);
        }

        $this->io->success('End');

        $memoryUsage = memory_get_usage(true) / 1024 / 1024;
        $this->io->text("Batch finished with memory: ${memoryUsage}M");

        return 0;
    }
}
