<?php

namespace App\Repository;

use App\Entity\Adherent;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @method Adherent|null find($id, $lockMode = null, $lockVersion = null)
 * @method Adherent|null findOneBy(array $criteria, array $orderBy = null)
 * @method Adherent[]    findAll()
 * @method Adherent[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class AdherentRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Adherent::class);
    }

    public function findOneByEmail($email, $enabledOnly = true)
    {
        $qb = $this->createQueryBuilder('p');

        if ($enabledOnly) {
            return $qb
                ->leftjoin('p.user', 'u')
                ->where('p.enabled = :enabled')
                ->andWhere('u.email = :email')
                ->setParameter('enabled', true)
                ->setParameter('email', $email)
                ->getQuery()
                ->getSingleResult()
            ;
        } else {
            return $qb
                ->leftjoin('p.user', 'u')
                ->where('u.email = :email')
                ->setParameter('email', $email)
                ->getQuery()
                ->getSingleResult()
            ;
        }
    }

    /**
     * @return Adherent[] Returns an array of Adherent objects
     */
    public function findByData(array $data)
    {
        $qb = $this->createQueryBuilder('p');

        $qb = $qb->leftjoin('p.user', 'u');

        if (isset($data['email'])) {
            $qb =
                $qb
                    ->orWhere('u.email = :email')
                    ->setParameter('email', $data['email'])
                ;
        }
        if (isset($data['name'])) {
            $qb =
                $qb
                    ->orWhere('u.firstname = :firstname')
                    ->orWhere('u.lastname = :firstname')
                    ->setParameter('email', $data['name'])
                ;
        }

        return $qb
            ->getQuery()
            ->getOneOrNullResult()
        ;
    }

    public function findsByData(array $data)
    {
        $qb = $this->createQueryBuilder('p');

        $qb = $qb
            ->leftjoin('p.user', 'u')
        ;

        if (isset($data['email'])) {
            $qb =
                $qb
                    ->orWhere('u.email = :email')
                    ->setParameter('email', $data['email'])
                ;
        }
        if (isset($data['name'])) {
            $qb =
                $qb
                    ->orWhere('CONCAT(u.firstname, \' \', u.lastname) = :name')
                    ->orWhere('CONCAT(u.lastname, \' \', u.firstname) = :name')
                    ->setParameter('name', $data['name'])
                ;
        }

        return $qb
            ->andWhere('p.enabled = :enabled')
            ->setParameter('enabled', true)
            ->orderBy('u.lastname', 'ASC')
            ->getQuery()
            ->getResult()
        ;
    }

    public function findOrderByName()
    {
        $qb = $this->createQueryBuilder('p');

        return $qb
            ->leftjoin('p.user', 'u')
            ->where('p.enabled = :enabled')
            ->setParameter('enabled', true)
            ->orderBy('u.lastname', 'ASC')
            ->getQuery()
            ->getResult()
        ;
    }

    public function findOrderByNameExcludeInactiveUser()
    {
        $qb = $this->createQueryBuilder('p');

        return $qb
            ->leftjoin('p.user', 'u')
            ->where('p.enabled = :enabled')
            ->andWhere('u.enabled = :userEnabled')
            ->setParameters(['enabled' => true, 'userEnabled' => true])
            ->orderBy('u.lastname', 'ASC')
            ->getQuery()
            ->getResult()
            ;
    }

    /**
     * @return Adherent[] Returns an array of Adherent objects
     */
    public function findbyExclude(Adherent $adherent)
    {
        $qb = $this->createQueryBuilder('p');

        return $qb
            ->leftjoin('p.user', 'u')
            ->where('p.id != :presta')
            ->andWhere('p.enabled = :enabled')
            ->setParameter('presta', $adherent->getId())
            ->setParameter('enabled', true)
            ->orderBy('u.lastname', 'ASC')
            ->getQuery()
            ->getResult()
        ;
    }
}
