<?php

declare(strict_types=1);

namespace App\Command;

use App\Entity\Adherent;
use App\Entity\GlobalParameter;
use App\Utils\CustomEntityManager;
use App\Utils\TAVCotisationUtils;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Twig\Environment;
use Symfony\Component\Console\Input\InputArgument;

/**
 * This command is used to import data to complete adherents profile,
 * in order to insure a transition towards simplified household based allowance process.
 */
class ImportSimplifiedHouseholdAdherentsData extends Command
{
    protected static $defaultName = 'kohinos:ssa:simplified-household-import';

    protected $em;
    protected $io;
    protected $param;
    protected $tavCotisationUtils;

    public function __construct(
        CustomEntityManager $em,
        TAVCotisationUtils $tavCotisationUtils
    ) {
        $this->em = $em;
        $this->tavCotisationUtils = $tavCotisationUtils;
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setDescription('SSA : importer via CSV les informations des adhérents existants pour l\'allocation par foyer simplifiée')
            ->addArgument('filepath', InputArgument::REQUIRED, 'Path to the csv file')
        ;
    }

    private function readCsv() {

    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->io = new SymfonyStyle($input, $output);

        $this->io->title('START - importing simplified household data');

        $csvFilePath = $input->getArgument('filepath');

        if (($handle = fopen($csvFilePath, "r")) !== FALSE) {
            while (($row = fgetcsv($handle, 1000, ",")) !== FALSE) {
                if ($row[0] === 'email') {
                    // headers row
                    continue;
                }

                try {
                    $adherent = $this->em->getRepository(Adherent::class)->findOneByEmail($row[0], false);
                } catch (\Exception $e) {
                    $this->io->warning('Adherent not found: ' . $row[0]);
                    continue;
                }

                $updated = false;

                if (is_null($adherent->getHouseholdCount())) {
                    $adherent->setHouseholdCount(intval($row[1]));
                    $updated = true;
                }

                if (is_null($adherent->getCotisationAmount())) {
                    $adherent->setCotisationAmount(intval($row[2]));
                    $updated = true;
                }

                if ($updated) {
                    $this->tavCotisationUtils->calculateAllowanceAccordingToHouseholdSimplified($adherent);
                    $this->em->persist($adherent);
                    $this->io->success('Succesfully updated: ' . $row[0]);
                } else {
                    $this->io->warning('Profil already complete for Adherent: ' . $row[0] . ' (update ignored)');
                }
            }

            $this->em->flush();
            fclose($handle);
        }

        $this->io->success('End');

        $memoryUsage = memory_get_usage(true) / 1024 / 1024;
        $this->io->text("Batch finished with memory: ${memoryUsage}M");

        return 0;
    }
}
