<?php

namespace App\Entity;

use App\Entity\EntityTrait\GeolocEntityTrait;
use App\Repository\ThirdPartyFinancerRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;

/**
 * Third party entities that participate to fund adherent allocations.
 * Each Adhérent can only be supported by one Third Party Financer.
 * Each time an adherent gets his allowance :
 *  - the flux of cosisation and reversement are created as usual 
 *  - create a ThirdPartyAllocationFunding instance to save the information that the financer supported funded [fundingAmount] among the reversement amount
 * 
 * @ORM\Entity(repositoryClass=ThirdPartyFinancerRepository::class)
 */
class ThirdPartyFinancer
{
    use GeolocEntityTrait;

    /**
     * @ORM\Id
     * @ORM\GeneratedValue
     * @ORM\Column(type="integer")
     */
    private $id;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private $name;

    /**
     * @ORM\Column(type="float")
     */
    private $fundingAmount;

    /**
     * @ORM\OneToMany(targetEntity=Adherent::class, mappedBy="thirdPartyFinancer")
     */
    private $supportedAdherents;

    /**
     * @ORM\OneToMany(targetEntity=ThirdPartyAllocationFunding::class, mappedBy="thirdPartyFinancer", orphanRemoval=true)
     */
    private $thirdPartyAllocationFundings;

    public function __construct()
    {
        $this->supportedAdherents = new ArrayCollection();
        $this->thirdPartyAllocationFundings = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(string $name): self
    {
        $this->name = $name;

        return $this;
    }

    public function getFundingAmount(): ?float
    {
        return $this->fundingAmount;
    }

    public function setFundingAmount(float $fundingAmount): self
    {
        $this->fundingAmount = $fundingAmount;

        return $this;
    }

    /**
     * @return Collection<int, Adherent>
     */
    public function getSupportedAdherents(): Collection
    {
        return $this->supportedAdherents;
    }

    public function addSupportedAdherent(Adherent $supportedAdherent): self
    {
        if (!$this->supportedAdherents->contains($supportedAdherent)) {
            $this->supportedAdherents[] = $supportedAdherent;
            $supportedAdherent->setThirdPartyFinancer($this);
        }

        return $this;
    }

    public function removeSupportedAdherent(Adherent $supportedAdherent): self
    {
        if ($this->supportedAdherents->removeElement($supportedAdherent)) {
            // set the owning side to null (unless already changed)
            if ($supportedAdherent->getThirdPartyFinancer() === $this) {
                $supportedAdherent->setThirdPartyFinancer(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, ThirdPartyAllocationFunding>
     */
    public function getThirdPartyAllocationFundings(): Collection
    {
        return $this->thirdPartyAllocationFundings;
    }

    public function addThirdPartyAllocationFunding(ThirdPartyAllocationFunding $thirdPartyAllocationFunding): self
    {
        if (!$this->thirdPartyAllocationFundings->contains($thirdPartyAllocationFunding)) {
            $this->thirdPartyAllocationFundings[] = $thirdPartyAllocationFunding;
            $thirdPartyAllocationFunding->setThirdPartyFinancer($this);
        }

        return $this;
    }

    public function removeThirdPartyAllocationFunding(ThirdPartyAllocationFunding $thirdPartyAllocationFunding): self
    {
        if ($this->thirdPartyAllocationFundings->removeElement($thirdPartyAllocationFunding)) {
            // set the owning side to null (unless already changed)
            if ($thirdPartyAllocationFunding->getThirdPartyFinancer() === $this) {
                $thirdPartyAllocationFunding->setThirdPartyFinancer(null);
            }
        }

        return $this;
    }

    public function __toString(): string
    {
        return $this->getName();
    }
}
